# Loan System Migration Guide

## Overview
This guide explains how to migrate from the standard `EmployeeLoanController` to the improved `ImprovedEmployeeLoanController` with better architecture and SWIFT code support.

## What's Improved

### 1. **Better Architecture**
- **Service Layer**: Separated business logic into services
- **Repository Pattern**: Data access through repositories
- **Normalized Database**: Proper relational structure with audit trails
- **SWIFT Code Support**: Automatic bank identification and channel selection

### 2. **Enhanced Features**
- Automatic channel determination (INTERNAL, DOMESTIC, TISS, MNO)
- Proper SWIFT code mapping for 45+ Tanzania banks
- Better disbursement tracking
- Improved error handling and logging

## Migration Steps

### Step 1: Run Migrations
```bash
php artisan migrate
```

### Step 2: Seed Banks Data
```bash
php artisan db:seed --class=BankSwiftCodeSeeder
```

### Step 3: Update .env File
Add these configuration values to your `.env` file:
```env
# Enable improved controllers
USE_IMPROVED_LOAN_CONTROLLER=true
USE_IMPROVED_DISBURSEMENT=true
USE_IMPROVED_NMB_SERVICE=true

# ESS Configuration (if not already present)
ESS_PUBLIC_KEY_PATH=/home/crm/ess_utumishi_go_tz.crt
ESS_FSP_CODE=FL7456
ESS_SENDER_NAME="URA SACCOS LTD LOAN"
ESS_RECEIVER_NAME=ESS_UTUMISHI
```

### Step 4: Clear Cache
```bash
php artisan config:clear
php artisan cache:clear
php artisan route:clear
```

### Step 5: Test the New Endpoints

#### ESS Integration Endpoint
The improved controller provides a new endpoint for ESS:
- **Old**: `POST /api/employee_loan`
- **New**: `POST /api/employee_loan_v2`

You can run both in parallel during migration.

#### Web Interface
Access the improved interface at:
- Loan Offers: `/v2/loan-offers`
- Disbursements: `/v2/disbursements`

## Testing Strategy

### 1. Parallel Testing
Run both controllers simultaneously:
```env
# In .env for testing
USE_IMPROVED_LOAN_CONTROLLER=false  # Keep using old controller
```

Then manually test the new endpoints at `/api/employee_loan_v2`

### 2. Gradual Migration
1. Start with read-only operations (viewing loans)
2. Test loan calculations
3. Test approvals
4. Test disbursements
5. Switch production traffic

### 3. Rollback Plan
To rollback, simply change in `.env`:
```env
USE_IMPROVED_LOAN_CONTROLLER=false
USE_IMPROVED_DISBURSEMENT=false
```

## Database Changes

### New Tables Created
1. **banks** - Stores bank information with SWIFT codes
2. **bank_branches** - Bank branch details
3. **loan_offer_approvals** - Tracks all approval stages
4. **loan_disbursements** - Disbursement records with channel tracking
5. **loan_offer_topups** - Top-up loan relationships

### Modified Tables
- **loan_offers** - Added `bank_id` and `loan_type` fields
- **loan_disbursements** - Added channel tracking fields

## API Compatibility

The improved system maintains **100% ESS API compatibility**:
- Same XML message formats
- Same signature verification
- Same response structures
- All message types supported

## Channel Selection Logic

The system automatically selects the correct channel:

| Destination | Amount | Channel | Description |
|------------|--------|---------|-------------|
| NMB to NMB | Any | INTERNAL | Internal NMB transfer |
| Other Banks | < 20M TZS | DOMESTIC | Domestic transfer |
| Other Banks | ≥ 20M TZS | TISS | Tanzania Interbank Settlement System |
| Mobile Money | Any | MNO | Mobile Network Operator |

## SWIFT Code Mapping

All 45 Tanzania banks are mapped with their SWIFT codes:
- NMB: `NMIBTZTZ`
- CRDB: `CORUTZTZ`
- NBC: `NLCBTZTX`
- And 42 more...

## Monitoring

### Check Which Controller is Active
Look in Laravel logs:
```
[2024-08-31] Using ImprovedEmployeeLoanController for ESS integration
```

### Track Disbursements
Query the `loan_disbursements` table:
```sql
SELECT 
    channel_identifier,
    destination_code,
    swift_code,
    status,
    COUNT(*) as count
FROM loan_disbursements
GROUP BY channel_identifier, destination_code, swift_code, status;
```

## Troubleshooting

### Issue: Routes Not Found
**Solution**: Clear route cache
```bash
php artisan route:clear
php artisan route:cache
```

### Issue: Banks Not Found
**Solution**: Run seeder
```bash
php artisan db:seed --class=BankSwiftCodeSeeder
```

### Issue: Old Controller Still Being Used
**Solution**: Check `.env` and clear config cache
```bash
php artisan config:clear
```

## Benefits After Migration

1. **Better Performance**: Optimized queries with proper indexes
2. **Audit Trail**: Complete history of approvals and disbursements
3. **Accurate Disbursements**: Automatic channel and destination selection
4. **Maintainability**: Clean code structure with separation of concerns
5. **Testability**: Services can be unit tested independently
6. **Scalability**: Ready for future enhancements

## Support

For issues or questions:
1. Check Laravel logs: `storage/logs/laravel.log`
2. Verify database migrations: `php artisan migrate:status`
3. Test ESS connectivity with test payloads
4. Review disbursement records in `loan_disbursements` table

## Next Steps

After successful migration:
1. Update ESS to use the new endpoint
2. Train staff on new features
3. Set up monitoring dashboards
4. Document any customizations
5. Plan for regular database maintenance

---

**Note**: Keep the old controller files until you're confident the migration is complete and stable.