<?php
/**
 * Script to verify banks table data and sync with loan offers
 */

require_once __DIR__ . '/vendor/autoload.php';
$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
$response = $kernel->handle(
    $request = Illuminate\Http\Request::capture()
);

use App\Models\Bank;
use App\Models\LoanOffer;
use Illuminate\Support\Facades\DB;

echo "\n===========================================\n";
echo "BANKS TABLE DATA VERIFICATION\n";
echo "===========================================\n\n";

// 1. Check banks table
$banksCount = Bank::count();
echo "1. Banks in database: {$banksCount}\n";

if ($banksCount > 0) {
    echo "\nSample banks with SWIFT codes:\n";
    echo str_pad("Bank Name", 40) . str_pad("Short Name", 15) . str_pad("SWIFT Code", 15) . "\n";
    echo str_repeat("-", 70) . "\n";
    
    Bank::limit(10)->get()->each(function($bank) {
        echo str_pad(substr($bank->name, 0, 39), 40);
        echo str_pad($bank->short_name ?: 'N/A', 15);
        echo str_pad($bank->swift_code, 15) . "\n";
    });
}

// 2. Check loan offers with SWIFT codes but no bank_id
echo "\n2. Checking loan offers with SWIFT codes...\n";
$loansWithSwift = LoanOffer::whereNotNull('swift_code')->count();
$loansWithBankId = LoanOffer::whereNotNull('bank_id')->count();
$loansWithSwiftNoBank = LoanOffer::whereNotNull('swift_code')->whereNull('bank_id')->count();

echo "   - Total loans with SWIFT code: {$loansWithSwift}\n";
echo "   - Loans linked to banks table: {$loansWithBankId}\n";
echo "   - Loans with SWIFT but no bank link: {$loansWithSwiftNoBank}\n";

// 3. Find unique SWIFT codes in loan_offers not in banks table
echo "\n3. SWIFT codes in loan_offers not found in banks table:\n";
$unmatchedSwiftCodes = DB::table('loan_offers')
    ->whereNotNull('swift_code')
    ->whereNull('bank_id')
    ->select('swift_code')
    ->distinct()
    ->pluck('swift_code');

if ($unmatchedSwiftCodes->count() > 0) {
    echo "   Found {$unmatchedSwiftCodes->count()} unmatched SWIFT codes:\n";
    foreach ($unmatchedSwiftCodes as $code) {
        $count = LoanOffer::where('swift_code', $code)->count();
        echo "   - {$code} (used in {$count} loan offers)\n";
    }
    
    echo "\n4. Suggested banks to add:\n";
    echo "   Copy this to your BankSwiftCodeSeeder or add manually:\n\n";
    
    foreach ($unmatchedSwiftCodes as $code) {
        echo "   Bank::create([\n";
        echo "       'swift_code' => '{$code}',\n";
        echo "       'name' => 'BANK_NAME_HERE', // TODO: Add actual bank name\n";
        echo "       'short_name' => 'SHORT_NAME', // TODO: Add short name\n";
        echo "   ]);\n\n";
    }
} else {
    echo "   ✓ All SWIFT codes in loan_offers have matching banks!\n";
}

// 5. Attempt to link unlinked loans
echo "\n5. Attempting to link loans with matching SWIFT codes...\n";
$updated = 0;
$loansToUpdate = LoanOffer::whereNotNull('swift_code')->whereNull('bank_id')->get();

foreach ($loansToUpdate as $loan) {
    $bank = Bank::where('swift_code', $loan->swift_code)->first();
    if ($bank) {
        $loan->bank_id = $bank->id;
        $loan->save();
        $updated++;
    }
}

if ($updated > 0) {
    echo "   ✓ Successfully linked {$updated} loan offers to banks!\n";
} else {
    echo "   No loans to update or no matching banks found.\n";
}

// 6. Summary
echo "\n===========================================\n";
echo "SUMMARY\n";
echo "===========================================\n";
echo "Banks in system: {$banksCount}\n";
echo "Loans with banks linked: " . LoanOffer::whereNotNull('bank_id')->count() . "\n";
echo "Loans needing bank link: " . LoanOffer::whereNotNull('swift_code')->whereNull('bank_id')->count() . "\n";

if ($unmatchedSwiftCodes->count() > 0) {
    echo "\nACTION REQUIRED:\n";
    echo "1. Add the missing banks to the banks table using the suggested code above\n";
    echo "2. Run this script again to link the loans\n";
    echo "3. Or run: php artisan db:seed --class=BankSwiftCodeSeeder\n";
} else {
    echo "\n✓ All systems operational! Banks are properly linked.\n";
}

echo "\n";