@extends('layouts.app')

@section('content')
<div class="container-fluid py-4 bg-gradient">
    <!-- Enhanced Page Header with Progress Indicator -->
    <div class="page-header-enhanced mb-4">
        <div class="row align-items-center">
            <div class="col-lg-7">
                <nav aria-label="breadcrumb">
                    <ol class="breadcrumb bg-transparent px-0 mb-2">
                        <li class="breadcrumb-item"><a href="{{ route('dashboard') }}"><i class="fas fa-home"></i></a></li>
                        <li class="breadcrumb-item"><a href="{{ route('loan-offers.index') }}">Loans</a></li>
                        <li class="breadcrumb-item active">{{ $loanOffer->application_number }}</li>
                    </ol>
                </nav>
                <div class="d-flex align-items-center">
                    <div class="page-icon me-3">
                        <i class="fas fa-file-invoice-dollar"></i>
                    </div>
                    <div>
                        <h1 class="page-title mb-0">Loan Application Details</h1>
                        <p class="text-muted mb-0">Complete loan information and management</p>
                    </div>
                </div>
            </div>
            <div class="col-lg-5 text-lg-end mt-3 mt-lg-0">
                <div class="btn-toolbar justify-content-lg-end">
                    <div class="btn-group me-2">
                        <button class="btn btn-light" onclick="window.print()">
                            <i class="fas fa-print"></i>
                        </button>
                        <button class="btn btn-light" data-bs-toggle="dropdown">
                            <i class="fas fa-share"></i>
                        </button>
                        <ul class="dropdown-menu">
                            <li><a class="dropdown-item" href="#"><i class="fas fa-envelope me-2"></i>Email</a></li>
                            <li><a class="dropdown-item" href="#"><i class="fas fa-file-pdf me-2"></i>Export PDF</a></li>
                        </ul>
                    </div>
                    <a href="{{ route('loan-offers.index') }}" class="btn btn-outline-secondary">
                        <i class="fas fa-arrow-left me-2"></i>Back
                    </a>
                </div>
            </div>
        </div>
    </div>

    <!-- Loan Progress Timeline -->
    <div class="loan-progress-bar mb-4">
        <div class="progress-wrapper">
            <div class="progress" style="height: 6px;">
                <div class="progress-bar" role="progressbar" style="background: linear-gradient(90deg, var(--ura-primary) 0%, var(--ura-secondary) 100%);" 
                     style="width: {{ $loanOffer->status == 'disbursed' ? '100' : ($loanOffer->approval == 'APPROVED' ? '66' : '33') }}%">
                </div>
            </div>
            <div class="progress-steps">
                <div class="step {{ $loanOffer->created_at ? 'completed' : '' }}">
                    <div class="step-icon">
                        <i class="fas fa-file-alt"></i>
                    </div>
                    <span>Applied</span>
                </div>
                <div class="step {{ $loanOffer->approval == 'APPROVED' ? 'completed' : ($loanOffer->approval == 'REJECTED' ? 'rejected' : '') }}">
                    <div class="step-icon">
                        <i class="fas {{ $loanOffer->approval == 'REJECTED' ? 'fa-times' : 'fa-check' }}"></i>
                    </div>
                    <span>{{ $loanOffer->approval ?: 'Review' }}</span>
                </div>
                <div class="step {{ $loanOffer->status == 'disbursed' ? 'completed' : '' }}">
                    <div class="step-icon">
                        <i class="fas fa-hand-holding-usd"></i>
                    </div>
                    <span>Disbursed</span>
                </div>
            </div>
        </div>
    </div>

    <!-- Enhanced Applicant Card with More Details -->
    <div class="applicant-detail-card mb-4">
        <div class="card-body">
            <div class="row">
                <div class="col-lg-8">
                    <div class="d-flex align-items-start">
                        <div class="avatar-xlarge me-4">
                            <span class="avatar-text">
                                {{ strtoupper(substr($loanOffer->first_name, 0, 1)) }}{{ strtoupper(substr($loanOffer->last_name, 0, 1)) }}
                            </span>
                            <span class="avatar-status {{ $loanOffer->approval == 'APPROVED' ? 'bg-success' : 'bg-warning' }}"></span>
                        </div>
                        <div class="flex-grow-1">
                            <h2 class="mb-1">
                                {{ $loanOffer->first_name }} {{ $loanOffer->middle_name }} {{ $loanOffer->last_name }}
                                @if($loanOffer->sex)
                                    <span class="badge bg-light text-dark ms-2">{{ $loanOffer->sex }}</span>
                                @endif
                            </h2>
                            <p class="text-muted mb-3">{{ $loanOffer->designation_name }} ({{ $loanOffer->designation_code }})</p>
                            
                            <div class="row g-3">
                                <div class="col-md-6">
                                    <div class="info-item">
                                        <i class="fas fa-id-card text-primary"></i>
                                        <span class="label">Check Number:</span>
                                        <strong>{{ $loanOffer->check_number }}</strong>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="info-item">
                                        <i class="fas fa-fingerprint text-primary"></i>
                                        <span class="label">NIN:</span>
                                        <strong>{{ $loanOffer->nin ?: 'Not provided' }}</strong>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="info-item">
                                        <i class="fas fa-ring text-primary"></i>
                                        <span class="label">Marital Status:</span>
                                        <strong>{{ $loanOffer->marital_status ?: 'Not specified' }}</strong>
                                    </div>
                                </div>
                                <div class="col-md-6">
                                    <div class="info-item">
                                        <i class="fas fa-briefcase text-primary"></i>
                                        <span class="label">Employment:</span>
                                        <strong>{{ $loanOffer->terms_of_employment ?: 'Permanent' }}</strong>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-4">
                    <div class="loan-amount-display">
                        <small class="text-muted">Loan Amount</small>
                        <h3 class="mb-2">TZS {{ number_format($loanOffer->total_amount_to_pay, 2) }}</h3>
                        <div class="loan-breakdown">
                            <div class="d-flex justify-content-between mb-1">
                                <span>Principal:</span>
                                <strong>TZS {{ number_format($loanOffer->requested_amount, 2) }}</strong>
                            </div>
                            <div class="d-flex justify-content-between mb-1">
                                <span>Interest ({{ $loanOffer->interest_rate }}%):</span>
                                <strong>TZS {{ number_format(($loanOffer->total_amount_to_pay - $loanOffer->requested_amount), 2) }}</strong>
                            </div>
                            @if($loanOffer->other_charges > 0)
                            <div class="d-flex justify-content-between">
                                <span>Other Charges:</span>
                                <strong>TZS {{ number_format($loanOffer->other_charges, 2) }}</strong>
                            </div>
                            @endif
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <!-- Quick Action Cards -->
    <div class="row g-3 mb-4">
        @if($loanOffer->approval === 'APPROVED' && !in_array($loanOffer->status, ['disbursement_pending', 'disbursed', 'FULL_SETTLED', 'DISBURSEMENT_FAILED']))
        <div class="col-12">
            <div class="action-alert-card ready-to-disburse">
                <div class="row align-items-center">
                    <div class="col-lg-8">
                        <div class="d-flex align-items-center">
                            <div class="alert-icon pulse">
                                <i class="fas fa-exclamation-circle"></i>
                            </div>
                            <div>
                                <h5 class="mb-1">Ready for Disbursement</h5>
                                <p class="mb-0">This loan is approved and awaiting disbursement to NMB Bank</p>
                            </div>
                        </div>
                    </div>
                    <div class="col-lg-4 text-lg-end mt-3 mt-lg-0">
                        <button id="disburse-btn" class="btn btn-success btn-lg px-4">
                            <i class="fas fa-paper-plane me-2"></i>Disburse Now
                        </button>
                    </div>
                </div>
            </div>
        </div>
        @endif
    </div>

    <!-- Main Content Tabs -->
    <div class="card border-0 shadow-sm">
        <div class="card-header bg-white border-0 pt-4">
            <ul class="nav nav-tabs-custom" role="tablist">
                <li class="nav-item">
                    <a class="nav-link active" data-bs-toggle="tab" href="#loan-details">
                        <i class="fas fa-file-invoice me-2"></i>Loan Details
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#personal-info">
                        <i class="fas fa-user me-2"></i>Personal Info
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#employment-info">
                        <i class="fas fa-briefcase me-2"></i>Employment
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#financial-info">
                        <i class="fas fa-chart-line me-2"></i>Financial
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#contact-info">
                        <i class="fas fa-address-book me-2"></i>Contact
                    </a>
                </li>
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#payment-info">
                        <i class="fas fa-credit-card me-2"></i>Payment
                    </a>
                </li>
                @if($loanOffer->nmb_batch_id || $loanOffer->callbacks->count() > 0)
                <li class="nav-item">
                    <a class="nav-link" data-bs-toggle="tab" href="#activity-log">
                        <i class="fas fa-history me-2"></i>Activity
                        <span class="badge bg-danger ms-1">{{ $loanOffer->callbacks->count() }}</span>
                    </a>
                </li>
                @endif
            </ul>
        </div>
        
        <div class="card-body p-4">
            <form id="loan-update-form" novalidate>
                @csrf
                @method('PUT')
                
                <div class="tab-content">
                    <!-- Loan Details Tab -->
                    <div class="tab-pane fade show active" id="loan-details">
                        <div class="row g-4">
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Application Information</h6>
                                    <div class="detail-list">
                                        <div class="detail-item">
                                            <span class="detail-label">Application Number:</span>
                                            <span class="detail-value">{{ $loanOffer->application_number }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">FSP Reference:</span>
                                            <span class="detail-value">{{ $loanOffer->fsp_reference_number ?: 'Pending' }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Loan Number:</span>
                                            <span class="detail-value">{{ $loanOffer->loan_number ?: 'Not assigned' }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Product Code:</span>
                                            <span class="detail-value">{{ $loanOffer->product_code }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Loan Purpose:</span>
                                            <span class="detail-value">{{ $loanOffer->loan_purpose ?: 'Personal use' }}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Loan Terms</h6>
                                    <div class="detail-list">
                                        <div class="detail-item">
                                            <span class="detail-label">Requested Amount:</span>
                                            <span class="detail-value fw-bold">TZS {{ number_format($loanOffer->requested_amount, 2) }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Interest Rate:</span>
                                            <span class="detail-value">{{ $loanOffer->interest_rate }}% per annum</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Tenure:</span>
                                            <span class="detail-value">{{ $loanOffer->tenure }} months</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Monthly Deduction:</span>
                                            <span class="detail-value fw-bold">TZS {{ number_format($loanOffer->desired_deductible_amount, 2) }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Total Payable:</span>
                                            <span class="detail-value fw-bold text-primary">TZS {{ number_format($loanOffer->total_amount_to_pay, 2) }}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Fees & Charges</h6>
                                    <div class="detail-list">
                                        <div class="detail-item">
                                            <span class="detail-label">Processing Fee:</span>
                                            <span class="detail-value">TZS {{ number_format($loanOffer->processing_fee, 2) }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Insurance:</span>
                                            <span class="detail-value">TZS {{ number_format($loanOffer->insurance, 2) }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Other Charges:</span>
                                            <span class="detail-value">TZS {{ number_format($loanOffer->other_charges, 2) }}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Important Dates</h6>
                                    <div class="detail-list">
                                        <div class="detail-item">
                                            <span class="detail-label">Application Date:</span>
                                            <span class="detail-value">{{ $loanOffer->created_at->format('d M Y') }}</span>
                                        </div>
                                        @if($loanOffer->contract_start_date)
                                        <div class="detail-item">
                                            <span class="detail-label">Contract Start:</span>
                                            <span class="detail-value">{{ $loanOffer->contract_start_date }}</span>
                                        </div>
                                        @endif
                                        @if($loanOffer->contract_end_date)
                                        <div class="detail-item">
                                            <span class="detail-label">Contract End:</span>
                                            <span class="detail-value">{{ $loanOffer->contract_end_date }}</span>
                                        </div>
                                        @endif
                                        <div class="detail-item">
                                            <span class="detail-label">Retirement Date:</span>
                                            <span class="detail-value">{{ $loanOffer->retirement_date ?: 'Not specified' }}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            <!-- Editable Status Section -->
                            <div class="col-12">
                                <div class="edit-section">
                                    <h6 class="detail-group-title">
                                        <i class="fas fa-edit text-primary me-2"></i>Status Management
                                    </h6>
                                    <fieldset @if(in_array($loanOffer->status, ['disbursement_pending', 'disbursed', 'FULL_SETTLED'])) disabled @endif>
                                        <div class="row g-3">
                                            <div class="col-md-4">
                                                <div class="form-floating">
                                                    <select class="form-select" id="approval" name="approval">
                                                        <option value="PENDING" {{ (!$loanOffer->approval || $loanOffer->approval === 'PENDING') ? 'selected' : '' }}>⏳ PENDING</option>
                                                        <option value="APPROVED" {{ $loanOffer->approval === 'APPROVED' ? 'selected' : '' }}>✅ APPROVED</option>
                                                        <option value="REJECTED" {{ $loanOffer->approval === 'REJECTED' ? 'selected' : '' }}>❌ REJECTED</option>
                                                    </select>
                                                    <label for="approval">Approval Status</label>
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="form-floating">
                                                    <input type="number" step="0.01" class="form-control" id="total_amount_to_pay" 
                                                           name="total_amount_to_pay" value="{{ $loanOffer->total_amount_to_pay }}">
                                                    <label for="total_amount_to_pay">Total Amount (TZS)</label>
                                                </div>
                                            </div>
                                            <div class="col-md-4">
                                                <div class="form-floating">
                                                    <input type="number" step="0.01" class="form-control" id="other_charges" 
                                                           name="other_charges" value="{{ $loanOffer->other_charges }}">
                                                    <label for="other_charges">Other Charges (TZS)</label>
                                                </div>
                                            </div>
                                            <div class="col-12">
                                                <div class="form-floating">
                                                    <textarea class="form-control" id="reason" name="reason" 
                                                              style="height: 80px">{{ $loanOffer->reason }}</textarea>
                                                    <label for="reason">Remarks / Reason</label>
                                                </div>
                                            </div>
                                        </div>
                                    </fieldset>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Personal Information Tab -->
                    <div class="tab-pane fade" id="personal-info">
                        <div class="row g-4">
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Basic Information</h6>
                                    <div class="detail-list">
                                        <div class="detail-item">
                                            <span class="detail-label">Full Name:</span>
                                            <span class="detail-value">{{ $loanOffer->first_name }} {{ $loanOffer->middle_name }} {{ $loanOffer->last_name }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Gender:</span>
                                            <span class="detail-value">{{ $loanOffer->sex ?: 'Not specified' }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">NIN:</span>
                                            <span class="detail-value">{{ $loanOffer->nin ?: 'Not provided' }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Marital Status:</span>
                                            <span class="detail-value">{{ $loanOffer->marital_status ?: 'Not specified' }}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Physical Address</h6>
                                    <div class="address-box">
                                        <i class="fas fa-map-marker-alt text-primary me-2"></i>
                                        {{ $loanOffer->physical_address ?: 'No address provided' }}
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Employment Information Tab -->
                    <div class="tab-pane fade" id="employment-info">
                        <div class="row g-4">
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Position Details</h6>
                                    <div class="detail-list">
                                        <div class="detail-item">
                                            <span class="detail-label">Check Number:</span>
                                            <span class="detail-value">{{ $loanOffer->check_number }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Designation:</span>
                                            <span class="detail-value">{{ $loanOffer->designation_name }} ({{ $loanOffer->designation_code }})</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Vote:</span>
                                            <span class="detail-value">{{ $loanOffer->vote_name }} ({{ $loanOffer->vote_code }})</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Terms:</span>
                                            <span class="detail-value">{{ $loanOffer->terms_of_employment ?: 'Permanent' }}</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Employment Dates</h6>
                                    <div class="detail-list">
                                        <div class="detail-item">
                                            <span class="detail-label">Employment Date:</span>
                                            <span class="detail-value">{{ $loanOffer->employment_date ?: 'Not specified' }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Confirmation Date:</span>
                                            <span class="detail-value">{{ $loanOffer->confirmation_date ?: 'Not specified' }}</span>
                                        </div>
                                        <div class="detail-item">
                                            <span class="detail-label">Retirement Date:</span>
                                            <span class="detail-value">{{ $loanOffer->retirement_date ?: 'Not specified' }}</span>
                                        </div>
                                        @if($loanOffer->employment_date && $loanOffer->retirement_date)
                                        <div class="detail-item">
                                            <span class="detail-label">Years to Retirement:</span>
                                            <span class="detail-value">
                                                {{ \Carbon\Carbon::parse($loanOffer->retirement_date)->diffInYears(\Carbon\Carbon::now()) }} years
                                            </span>
                                        </div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Financial Information Tab -->
                    <div class="tab-pane fade" id="financial-info">
                        <div class="row g-4">
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Salary Information</h6>
                                    <div class="salary-card">
                                        <div class="salary-item">
                                            <span>Basic Salary</span>
                                            <h5 class="text-primary">TZS {{ number_format($loanOffer->basic_salary, 2) }}</h5>
                                        </div>
                                        <div class="salary-item">
                                            <span>Net Salary</span>
                                            <h5 class="text-success">TZS {{ number_format($loanOffer->net_salary, 2) }}</h5>
                                        </div>
                                        <div class="salary-item">
                                            <span>1/3 of Basic</span>
                                            <h5>TZS {{ number_format($loanOffer->one_third_amount, 2) }}</h5>
                                        </div>
                                        <div class="salary-item">
                                            <span>Total Deductions</span>
                                            <h5 class="text-danger">TZS {{ number_format($loanOffer->total_employee_deduction, 2) }}</h5>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Loan Repayment Analysis</h6>
                                    <div class="repayment-chart">
                                        <canvas id="repaymentChart"></canvas>
                                    </div>
                                    <div class="mt-3">
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Monthly Payment:</span>
                                            <strong>TZS {{ number_format($loanOffer->desired_deductible_amount, 2) }}</strong>
                                        </div>
                                        <div class="d-flex justify-content-between mb-2">
                                            <span>Debt-to-Income Ratio:</span>
                                            <strong>{{ $loanOffer->net_salary > 0 ? round(($loanOffer->desired_deductible_amount / $loanOffer->net_salary) * 100, 1) : 0 }}%</strong>
                                        </div>
                                        @if($loanOffer->installments_paid)
                                        <div class="d-flex justify-content-between">
                                            <span>Installments Paid:</span>
                                            <strong>{{ $loanOffer->installments_paid }} / {{ $loanOffer->tenure }}</strong>
                                        </div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Contact Information Tab -->
                    <div class="tab-pane fade" id="contact-info">
                        <div class="row g-4">
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Contact Details</h6>
                                    <div class="contact-list">
                                        <div class="contact-item">
                                            <i class="fas fa-phone text-primary"></i>
                                            <div>
                                                <small>Phone Number</small>
                                                <p class="mb-0">{{ $loanOffer->telephone_number ?: 'Not provided' }}</p>
                                            </div>
                                        </div>
                                        <div class="contact-item">
                                            <i class="fas fa-mobile-alt text-primary"></i>
                                            <div>
                                                <small>Mobile Number</small>
                                                <p class="mb-0">{{ $loanOffer->mobile_number ?: 'Not provided' }}</p>
                                            </div>
                                        </div>
                                        <div class="contact-item">
                                            <i class="fas fa-envelope text-primary"></i>
                                            <div>
                                                <small>Email Address</small>
                                                <p class="mb-0">{{ $loanOffer->email_address ?: 'Not provided' }}</p>
                                            </div>
                                        </div>
                                        <div class="contact-item">
                                            <i class="fas fa-map-marker-alt text-primary"></i>
                                            <div>
                                                <small>Physical Address</small>
                                                <p class="mb-0">{{ $loanOffer->physical_address ?: 'Not provided' }}</p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Bank Branch</h6>
                                    <div class="branch-card">
                                        <div class="branch-icon">
                                            <i class="fas fa-university"></i>
                                        </div>
                                        <div class="branch-info">
                                            <h6>{{ $loanOffer->nearest_branch_name ?: 'Not specified' }}</h6>
                                            <p class="mb-1">Branch Code: {{ $loanOffer->nearest_branch_code ?: 'N/A' }}</p>
                                            @if($loanOffer->swift_code)
                                            <p class="mb-0">SWIFT: {{ $loanOffer->swift_code }}</p>
                                            @endif
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Payment Information Tab -->
                    <div class="tab-pane fade" id="payment-info">
                        <div class="row g-4">
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Account Details</h6>
                                    <div class="account-card">
                                        <div class="account-number">
                                            <i class="fas fa-credit-card me-2"></i>
                                            {{ $loanOffer->bank_account_number ?: 'Not provided' }}
                                        </div>
                                        @if($loanOffer->paymentDestination)
                                        <div class="mt-3">
                                            <small class="text-muted">Payment Destination</small>
                                            <p class="mb-0">{{ $loanOffer->paymentDestination->name }}</p>
                                        </div>
                                        @endif
                                    </div>
                                    
                                    <!-- Editable Payment Destination -->
                                    <div class="mt-3">
                                        <fieldset @if(in_array($loanOffer->status, ['disbursement_pending', 'disbursed', 'FULL_SETTLED'])) disabled @endif>
                                            <div class="row g-3">
                                                <div class="col-12">
                                                    <div class="form-floating">
                                                        <input type="text" class="form-control" id="bank_account_number" 
                                                               name="bank_account_number" value="{{ $loanOffer->bank_account_number }}">
                                                        <label for="bank_account_number">Account/Wallet Number</label>
                                                    </div>
                                                </div>
                                                <div class="col-12">
                                                    <div class="form-floating">
                                                        <select class="form-select" id="payment_destination_id" name="payment_destination_id">
                                                            <option value="">Select Destination...</option>
                                                            @if(isset($destinations['BANK']))
                                                                <optgroup label="Banks">
                                                                    @foreach($destinations['BANK'] as $destination)
                                                                        <option value="{{ $destination->id }}" 
                                                                                {{ $loanOffer->payment_destination_id == $destination->id ? 'selected' : '' }}>
                                                                            {{ $destination->name }}
                                                                        </option>
                                                                    @endforeach
                                                                </optgroup>
                                                            @endif
                                                            @if(isset($destinations['MNO']))
                                                                <optgroup label="Mobile Money">
                                                                    @foreach($destinations['MNO'] as $destination)
                                                                        <option value="{{ $destination->id }}" 
                                                                                {{ $loanOffer->payment_destination_id == $destination->id ? 'selected' : '' }}>
                                                                            {{ $destination->name }}
                                                                        </option>
                                                                    @endforeach
                                                                </optgroup>
                                                            @endif
                                                        </select>
                                                        <label for="payment_destination_id">Payment Destination</label>
                                                    </div>
                                                </div>
                                            </div>
                                        </fieldset>
                                    </div>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="detail-group">
                                    <h6 class="detail-group-title">Payment Status</h6>
                                    <div class="payment-status-card">
                                        @if($loanOffer->outstanding_balance)
                                        <div class="status-item">
                                            <span>Outstanding Balance:</span>
                                            <strong class="text-danger">TZS {{ number_format($loanOffer->outstanding_balance, 2) }}</strong>
                                        </div>
                                        @endif
                                        @if($loanOffer->settlement_amount)
                                        <div class="status-item">
                                            <span>Settlement Amount:</span>
                                            <strong>TZS {{ number_format($loanOffer->settlement_amount, 2) }}</strong>
                                        </div>
                                        @endif
                                        @if($loanOffer->payment_reference_number)
                                        <div class="status-item">
                                            <span>Payment Reference:</span>
                                            <strong>{{ $loanOffer->payment_reference_number }}</strong>
                                        </div>
                                        @endif
                                        @if($loanOffer->last_deduction_date)
                                        <div class="status-item">
                                            <span>Last Deduction:</span>
                                            <strong>{{ \Carbon\Carbon::parse($loanOffer->last_deduction_date)->format('d M Y') }}</strong>
                                        </div>
                                        @endif
                                        @if($loanOffer->final_payment_date)
                                        <div class="status-item">
                                            <span>Final Payment:</span>
                                            <strong>{{ \Carbon\Carbon::parse($loanOffer->final_payment_date)->format('d M Y') }}</strong>
                                        </div>
                                        @endif
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Activity Log Tab -->
                    @if($loanOffer->nmb_batch_id || $loanOffer->callbacks->count() > 0)
                    <div class="tab-pane fade" id="activity-log">
                        <div class="activity-timeline">
                            <div class="timeline-header">
                                <h6 class="mb-3">Transaction History</h6>
                                @if($loanOffer->nmb_batch_id)
                                <div class="batch-info">
                                    <span class="badge bg-info">Batch ID: {{ $loanOffer->nmb_batch_id }}</span>
                                </div>
                                @endif
                            </div>
                            <div class="timeline-body">
                                @forelse($loanOffer->callbacks as $callback)
                                <div class="timeline-event">
                                    <div class="timeline-date">
                                        {{ $callback->created_at->format('d M Y, h:i A') }}
                                    </div>
                                    <div class="timeline-content">
                                        <div class="event-badge {{ $callback->final_status == 'success' ? 'bg-success' : 'bg-danger' }}">
                                            <i class="fas {{ $callback->final_status == 'success' ? 'fa-check' : 'fa-times' }}"></i>
                                        </div>
                                        <div class="event-details">
                                            <h6>{{ ucfirst($callback->final_status) }}</h6>
                                            <p>{{ $callback->status_description }}</p>
                                            @if($callback->payment_reference)
                                            <small>Ref: {{ $callback->payment_reference }}</small>
                                            @endif
                                        </div>
                                    </div>
                                </div>
                                @empty
                                <p class="text-muted text-center">No transaction history available</p>
                                @endforelse
                            </div>
                        </div>
                    </div>
                    @endif
                </div>
            </form>

            <!-- Action Buttons -->
            @if(!in_array($loanOffer->status, ['disbursement_pending', 'disbursed', 'FULL_SETTLED']))
            <div class="form-actions mt-4">
                <div class="d-flex justify-content-between align-items-center">
                    <div>
                        <button type="button" id="save-btn" class="btn btn-gradient-primary btn-lg">
                            <i class="fas fa-save me-2"></i>Save Changes
                        </button>
                        <button type="button" class="btn btn-outline-secondary btn-lg ms-2" onclick="resetForm()">
                            <i class="fas fa-undo me-2"></i>Reset
                        </button>
                    </div>
                    <small class="text-muted">
                        <i class="fas fa-clock me-1"></i>Auto-save enabled
                    </small>
                </div>
            </div>
            @endif
        </div>
    </div>
</div>

<!-- Modals -->
<div class="modal fade" id="disbursementModal" tabindex="-1">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header border-0">
                <h5 class="modal-title">
                    <i class="fas fa-paper-plane text-success me-2"></i>Confirm Disbursement
                </h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
            </div>
            <div class="modal-body text-center">
                <div class="mb-3">
                    <i class="fas fa-university fa-4x text-primary"></i>
                </div>
                <h4>TZS {{ number_format($loanOffer->total_amount_to_pay, 2) }}</h4>
                <p class="text-muted">Will be sent to NMB Bank for processing</p>
                <div class="alert alert-warning">
                    <i class="fas fa-exclamation-triangle me-2"></i>
                    This action cannot be undone
                </div>
            </div>
            <div class="modal-footer border-0">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-success" id="confirm-disburse">
                    <i class="fas fa-check me-2"></i>Confirm Disbursement
                </button>
            </div>
        </div>
    </div>
</div>
@endsection

@push('styles')
<style>
/* URASACCOS Brand Colors */
:root {
    --ura-primary: #17479E;
    --ura-secondary: #1e8449;
    --ura-accent: #ff6b35;
    --ura-warning: #f39c12;
    --ura-danger: #e74c3c;
    --ura-info: #3498db;
    --ura-light: #f4f6f7;
    --ura-dark: #2c3e50;
}

/* Enhanced Styles with URASACCOS Branding */
.bg-gradient {
    background: linear-gradient(135deg, var(--ura-primary) 0%, var(--ura-secondary) 100%);
    min-height: 100vh;
    position: relative;
}

.bg-gradient::before {
    content: '';
    position: absolute;
    top: 0;
    left: 0;
    right: 0;
    height: 400px;
    background: linear-gradient(180deg, rgba(255,255,255,0) 0%, #f8f9fa 100%);
    z-index: 0;
}

.container-fluid {
    position: relative;
    z-index: 1;
}

/* Page Header Enhanced */
.page-header-enhanced {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.08);
}

.page-icon {
    width: 60px;
    height: 60px;
    border-radius: 16px;
    background: linear-gradient(135deg, var(--ura-primary) 0%, var(--ura-secondary) 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
}

.page-title {
    font-size: 2rem;
    font-weight: 700;
    background: linear-gradient(135deg, var(--ura-primary) 0%, var(--ura-secondary) 100%);
    -webkit-background-clip: text;
    -webkit-text-fill-color: transparent;
}

/* Progress Timeline */
.loan-progress-bar {
    background: white;
    border-radius: 20px;
    padding: 2rem;
    box-shadow: 0 5px 20px rgba(0, 0, 0, 0.05);
}

.progress-wrapper {
    position: relative;
}

.progress-steps {
    display: flex;
    justify-content: space-between;
    position: relative;
    margin-top: -18px;
}

.step {
    text-align: center;
    position: relative;
}

.step-icon {
    width: 36px;
    height: 36px;
    border-radius: 50%;
    background: white;
    border: 3px solid #e0e6ed;
    display: flex;
    align-items: center;
    justify-content: center;
    margin: 0 auto 10px;
    transition: all 0.3s ease;
}

.step.completed .step-icon {
    background: linear-gradient(135deg, var(--ura-secondary) 0%, #27ae60 100%);
    border-color: transparent;
    color: white;
}

.step.rejected .step-icon {
    background: linear-gradient(135deg, var(--ura-danger) 0%, var(--ura-warning) 100%);
    border-color: transparent;
    color: white;
}

/* Applicant Detail Card */
.applicant-detail-card {
    background: white;
    border-radius: 20px;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.08);
    overflow: hidden;
}

.avatar-xlarge {
    width: 100px;
    height: 100px;
    border-radius: 20px;
    background: linear-gradient(135deg, var(--ura-primary) 0%, var(--ura-secondary) 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    position: relative;
}

.avatar-text {
    color: white;
    font-size: 2rem;
    font-weight: 700;
}

.avatar-status {
    position: absolute;
    bottom: 5px;
    right: 5px;
    width: 24px;
    height: 24px;
    border-radius: 50%;
    border: 3px solid white;
}

.info-item {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 8px 0;
}

.info-item .label {
    color: #6c757d;
    font-size: 0.875rem;
}

.loan-amount-display {
    background: linear-gradient(135deg, rgba(0, 51, 102, 0.1) 0%, rgba(30, 132, 73, 0.1) 100%);
    border-radius: 16px;
    padding: 1.5rem;
    height: 100%;
}

.loan-breakdown {
    background: white;
    border-radius: 12px;
    padding: 1rem;
    font-size: 0.875rem;
}

/* Action Alert Card */
.action-alert-card {
    background: linear-gradient(135deg, rgba(35, 213, 171, 0.1) 0%, rgba(35, 164, 85, 0.1) 100%);
    border: 2px solid rgba(35, 164, 85, 0.2);
    border-radius: 20px;
    padding: 1.5rem;
}

.alert-icon {
    width: 50px;
    height: 50px;
    border-radius: 50%;
    background: white;
    display: flex;
    align-items: center;
    justify-content: center;
    color: #23a455;
    font-size: 1.5rem;
    margin-right: 1.5rem;
}

.pulse {
    animation: pulse 2s infinite;
}

@keyframes pulse {
    0% { transform: scale(1); }
    50% { transform: scale(1.1); }
    100% { transform: scale(1); }
}

/* Custom Tabs */
.nav-tabs-custom {
    border: none;
    gap: 10px;
}

.nav-tabs-custom .nav-link {
    border: none;
    background: #f8f9fa;
    border-radius: 12px;
    padding: 12px 20px;
    color: #6c757d;
    font-weight: 600;
    transition: all 0.3s ease;
}

.nav-tabs-custom .nav-link:hover {
    background: #e9ecef;
    transform: translateY(-2px);
}

.nav-tabs-custom .nav-link.active {
    background: linear-gradient(135deg, var(--ura-primary) 0%, var(--ura-secondary) 100%);
    color: white;
}

/* Detail Groups */
.detail-group {
    background: #f8f9fa;
    border-radius: 16px;
    padding: 1.5rem;
    height: 100%;
}

.detail-group-title {
    color: #495057;
    font-size: 0.875rem;
    text-transform: uppercase;
    letter-spacing: 0.5px;
    margin-bottom: 1rem;
    font-weight: 700;
}

.detail-list {
    display: flex;
    flex-direction: column;
    gap: 12px;
}

.detail-item {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 8px 0;
    border-bottom: 1px solid rgba(0, 0, 0, 0.05);
}

.detail-item:last-child {
    border-bottom: none;
}

.detail-label {
    color: #6c757d;
    font-size: 0.875rem;
}

.detail-value {
    font-weight: 700;
    color: #212529;
    font-size: 0.95rem;
}

/* Edit Section */
.edit-section {
    background: linear-gradient(135deg, rgba(102, 126, 234, 0.05) 0%, rgba(118, 75, 162, 0.05) 100%);
    border-radius: 16px;
    padding: 1.5rem;
    border: 2px dashed rgba(102, 126, 234, 0.2);
}

/* Salary Card */
.salary-card {
    display: grid;
    grid-template-columns: repeat(2, 1fr);
    gap: 1rem;
}

.salary-item {
    background: white;
    border-radius: 12px;
    padding: 1rem;
    text-align: center;
}

.salary-item span {
    display: block;
    color: #6c757d;
    font-size: 0.875rem;
    margin-bottom: 8px;
}

.salary-item h5 {
    margin: 0;
    font-weight: 700;
}

/* Contact List */
.contact-list {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.contact-item {
    display: flex;
    align-items: start;
    gap: 1rem;
    padding: 1rem;
    background: white;
    border-radius: 12px;
}

.contact-item i {
    font-size: 1.25rem;
    margin-top: 4px;
}

.contact-item small {
    color: #6c757d;
    font-size: 0.75rem;
}

/* Branch Card */
.branch-card {
    background: white;
    border-radius: 16px;
    padding: 1.5rem;
    display: flex;
    align-items: center;
    gap: 1.5rem;
}

.branch-icon {
    width: 60px;
    height: 60px;
    border-radius: 12px;
    background: linear-gradient(135deg, var(--ura-info) 0%, var(--ura-accent) 100%);
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    font-size: 1.5rem;
}

/* Account Card */
.account-card {
    background: linear-gradient(135deg, var(--ura-primary) 0%, var(--ura-secondary) 100%);
    color: white;
    border-radius: 16px;
    padding: 2rem;
}

.account-number {
    font-size: 1.25rem;
    font-weight: 700;
    letter-spacing: 1px;
}

/* Timeline Events */
.activity-timeline {
    position: relative;
}

.timeline-event {
    display: flex;
    gap: 1rem;
    margin-bottom: 2rem;
    position: relative;
}

.timeline-event::before {
    content: '';
    position: absolute;
    left: 24px;
    top: 40px;
    bottom: -20px;
    width: 2px;
    background: #e0e6ed;
}

.timeline-event:last-child::before {
    display: none;
}

.event-badge {
    width: 48px;
    height: 48px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    color: white;
    flex-shrink: 0;
}

.timeline-date {
    color: #6c757d;
    font-size: 0.875rem;
    min-width: 150px;
}

.event-details h6 {
    margin-bottom: 0.5rem;
}

.event-details p {
    color: #6c757d;
    margin-bottom: 0.5rem;
}

/* Form Actions */
.form-actions {
    background: white;
    border-radius: 16px;
    padding: 1.5rem;
    margin-top: 2rem;
    box-shadow: 0 5px 20px rgba(0, 0, 0, 0.05);
}

/* Gradient Button */
.btn-gradient-primary {
    background: linear-gradient(135deg, var(--ura-primary) 0%, var(--ura-secondary) 100%);
    color: white;
    border: none;
    border-radius: 12px;
    padding: 0.75rem 2rem;
    font-weight: 600;
    transition: all 0.3s ease;
}

.btn-gradient-primary:hover {
    transform: translateY(-2px);
    box-shadow: 0 10px 25px rgba(102, 126, 234, 0.3);
    color: white;
}

/* URASACCOS Brand Enhancements with #17479E */
.btn-outline-secondary:hover {
    background: #17479E;
    border-color: #17479E;
    color: white;
    box-shadow: 0 4px 15px rgba(23, 71, 158, 0.3);
}

.badge {
    background: linear-gradient(135deg, #17479E 0%, #2563c7 100%);
    padding: 6px 12px;
    font-weight: 600;
}

.text-primary {
    color: #17479E !important;
}

.text-success {
    color: var(--ura-secondary) !important;
}

.border-primary {
    border-color: #17479E !important;
}

.bg-primary {
    background: linear-gradient(135deg, #17479E 0%, #2563c7 100%) !important;
}

.bg-success {
    background: linear-gradient(135deg, var(--ura-secondary) 0%, #27ae60 100%) !important;
}

/* Add URASACCOS watermark */
body::before {
    content: 'URASACCOS';
    position: fixed;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%) rotate(-45deg);
    font-size: 10rem;
    font-weight: 900;
    color: rgba(23, 71, 158, 0.03);
    z-index: -1;
    pointer-events: none;
}

/* Enhanced card hover effects */
.card, .detail-group, .applicant-detail-card {
    transition: all 0.4s cubic-bezier(0.175, 0.885, 0.32, 1.275);
}

.card:hover, .detail-group:hover {
    box-shadow: 0 15px 40px rgba(23, 71, 158, 0.2) !important;
}

/* Animated borders on focus */
.form-control, .form-select {
    border: 2px solid rgba(23, 71, 158, 0.2);
    transition: all 0.3s ease;
}

.form-control:focus, .form-select:focus {
    border-color: #17479E;
    box-shadow: 0 0 0 0.25rem rgba(23, 71, 158, 0.25);
    transform: translateY(-2px);
}

/* Button animations */
.btn {
    position: relative;
    overflow: hidden;
    transition: all 0.3s ease;
}

.btn::before {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    width: 0;
    height: 0;
    border-radius: 50%;
    background: rgba(255, 255, 255, 0.3);
    transform: translate(-50%, -50%);
    transition: width 0.6s, height 0.6s;
}

.btn:hover::before {
    width: 300px;
    height: 300px;
}

/* Responsive */
@media (max-width: 768px) {
    .page-title {
        font-size: 1.5rem;
    }
    
    .avatar-xlarge {
        width: 80px;
        height: 80px;
    }
    
    .progress-steps {
        font-size: 0.75rem;
    }
    
    .salary-card {
        grid-template-columns: 1fr;
    }
}

/* Animations */
@keyframes fadeInUp {
    from {
        opacity: 0;
        transform: translateY(20px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.applicant-detail-card,
.card,
.detail-group {
    animation: fadeInUp 0.5s ease forwards;
}
</style>
@endpush

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js"></script>
<script>
$(document).ready(function() {
    // Initialize Chart for Repayment Analysis
    const ctx = document.getElementById('repaymentChart');
    if (ctx) {
        new Chart(ctx.getContext('2d'), {
            type: 'doughnut',
            data: {
                labels: ['Principal', 'Interest', 'Fees'],
                datasets: [{
                    data: [
                        {{ $loanOffer->requested_amount }},
                        {{ $loanOffer->total_amount_to_pay - $loanOffer->requested_amount - $loanOffer->processing_fee - $loanOffer->insurance }},
                        {{ $loanOffer->processing_fee + $loanOffer->insurance }}
                    ],
                    backgroundColor: [
                        '#17479E', // URASACCOS Primary
                        '#1e8449', // URASACCOS Secondary
                        '#ff6b35'  // URASACCOS Accent
                    ],
                    borderWidth: 0
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    legend: {
                        position: 'bottom'
                    }
                }
            }
        });
    }

    // Save button functionality
    $('#save-btn').on('click', function(e) {
        const button = $(this);
        const originalHtml = button.html();
        
        // Show loading state
        button.html('<span class="spinner-border spinner-border-sm me-2"></span>Saving...').prop('disabled', true);
        
        $.ajax({
            url: "{{ route('loan-offers.update', $loanOffer->id) }}",
            type: 'POST',
            data: $('#loan-update-form').serialize(),
            success: function(response) {
                toastr.success('Changes saved successfully!');
                setTimeout(() => location.reload(), 1500);
            },
            error: function(xhr) {
                toastr.error('Failed to save changes');
                button.html(originalHtml).prop('disabled', false);
            }
        });
    });

    // Disburse button
    $('#disburse-btn').on('click', function() {
        $('#disbursementModal').modal('show');
    });

    $('#confirm-disburse').on('click', function() {
        $.ajax({
            url: "{{ route('loan-offers.update', $loanOffer->id) }}",
            type: 'POST',
            data: {
                '_token': "{{ csrf_token() }}",
                '_method': 'PUT',
                'status': 'SUBMITTED_FOR_DISBURSEMENT'
            },
            success: function(response) {
                $('#disbursementModal').modal('hide');
                Swal.fire({
                    icon: 'success',
                    title: 'Disbursement Initiated!',
                    text: 'Loan has been submitted to NMB for processing.',
                    timer: 3000
                }).then(() => location.reload());
            },
            error: function(xhr) {
                Swal.fire({
                    icon: 'error',
                    title: 'Failed',
                    text: 'Could not process disbursement'
                });
            }
        });
    });

    // Auto-save functionality
    let saveTimeout;
    $('#loan-update-form input, #loan-update-form select, #loan-update-form textarea').on('change', function() {
        clearTimeout(saveTimeout);
        saveTimeout = setTimeout(() => {
            // Auto-save logic here
            console.log('Auto-saving...');
        }, 3000);
    });
});

function resetForm() {
    document.getElementById('loan-update-form').reset();
}
</script>
@endpush