@extends('layouts.app')

@section('content')
<!-- DEBUG: User Role: {{ auth()->user()->roles->pluck('name')->implode(', ') }} | Enquiries Count: {{ $enquiries->count() }} -->
<style>
    /* Modern Dashboard Styles */
    :root {
        --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        --success-gradient: linear-gradient(135deg, #00b09b 0%, #96c93d 100%);
        --warning-gradient: linear-gradient(135deg, #f4c430 0%, #ffa500 100%);
        --danger-gradient: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
        --info-gradient: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
        --dark-gradient: linear-gradient(135deg, #30475e 0%, #222831 100%);
        --card-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        --card-hover-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
    }

    /* Dashboard Container */
    .dashboard-container {
        background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
        min-height: calc(100vh - 60px);
        padding: 30px 0;
    }

    /* Dashboard Header */
    .dashboard-header {
        background: white;
        border-radius: 15px;
        padding: 25px 30px;
        margin-bottom: 30px;
        box-shadow: var(--card-shadow);
        position: relative;
        overflow: hidden;
    }

    .dashboard-header::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 4px;
        background: var(--primary-gradient);
    }

    .dashboard-title {
        font-size: 28px;
        font-weight: 700;
        color: #2d3748;
        margin-bottom: 10px;
        display: flex;
        align-items: center;
        gap: 15px;
    }

    .dashboard-icon {
        width: 45px;
        height: 45px;
        background: var(--primary-gradient);
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 22px;
    }

    .dashboard-subtitle {
        color: #718096;
        font-size: 14px;
    }

    /* KPI Cards Section */
    .kpi-section {
        margin-bottom: 40px;
    }

    .kpi-grid {
        display: grid;
        grid-template-columns: repeat(4, 1fr);
        gap: 20px;
        margin-bottom: 30px;
    }

    .kpi-card {
        background: white;
        border-radius: 12px;
        padding: 18px;
        box-shadow: var(--card-shadow);
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .kpi-card:hover {
        transform: translateY(-5px);
        box-shadow: var(--card-hover-shadow);
    }

    .kpi-card::after {
        content: '';
        position: absolute;
        top: -50%;
        right: -50%;
        width: 200%;
        height: 200%;
        background: radial-gradient(circle, rgba(255, 255, 255, 0.1) 0%, transparent 70%);
        animation: float 15s ease-in-out infinite;
    }

    @keyframes float {
        0%, 100% { transform: translate(0, 0) rotate(0deg); }
        33% { transform: translate(30px, -30px) rotate(120deg); }
        66% { transform: translate(-20px, 20px) rotate(240deg); }
    }

    .kpi-header {
        display: flex;
        align-items: flex-start;
        justify-content: space-between;
        margin-bottom: 15px;
    }

    .kpi-icon-wrapper {
        width: 45px;
        height: 45px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 20px;
        color: white;
    }

    .kpi-icon-wrapper.pending {
        background: var(--warning-gradient);
    }

    .kpi-icon-wrapper.in-progress {
        background: var(--info-gradient);
    }

    .kpi-icon-wrapper.overdue {
        background: var(--danger-gradient);
    }

    .kpi-icon-wrapper.completed {
        background: var(--success-gradient);
    }

    .kpi-trend {
        display: flex;
        align-items: center;
        gap: 3px;
        padding: 4px 8px;
        border-radius: 15px;
        font-size: 11px;
        font-weight: 600;
    }

    .kpi-trend.up {
        background: rgba(16, 185, 129, 0.1);
        color: #10b981;
    }

    .kpi-trend.down {
        background: rgba(239, 68, 68, 0.1);
        color: #ef4444;
    }

    .kpi-trend.stable {
        background: rgba(59, 130, 246, 0.1);
        color: #3b82f6;
    }

    .kpi-value {
        font-size: 28px;
        font-weight: 700;
        color: #2d3748;
        margin-bottom: 5px;
        position: relative;
        z-index: 1;
    }

    .kpi-label {
        font-size: 13px;
        color: #718096;
        font-weight: 500;
        margin-bottom: 12px;
    }

    .kpi-footer {
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding-top: 12px;
        border-top: 1px solid #e2e8f0;
    }

    .kpi-change {
        font-size: 11px;
        color: #718096;
    }

    .kpi-link {
        font-size: 11px;
        color: #667eea;
        text-decoration: none;
        font-weight: 600;
        transition: color 0.3s ease;
    }

    .kpi-link:hover {
        color: #764ba2;
    }

    /* Quick Actions */
    .quick-actions {
        display: flex;
        gap: 15px;
        margin-bottom: 30px;
        flex-wrap: wrap;
    }

    .action-btn {
        padding: 12px 24px;
        border-radius: 10px;
        font-size: 14px;
        font-weight: 600;
        text-decoration: none;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 10px;
        border: none;
        cursor: pointer;
        color: white;
        position: relative;
        overflow: hidden;
    }

    .action-btn::before {
        content: '';
        position: absolute;
        top: 50%;
        left: 50%;
        width: 0;
        height: 0;
        background: rgba(255, 255, 255, 0.2);
        border-radius: 50%;
        transform: translate(-50%, -50%);
        transition: width 0.6s, height 0.6s;
    }

    .action-btn:hover::before {
        width: 300px;
        height: 300px;
    }

    .action-btn.primary {
        background: var(--primary-gradient);
    }

    .action-btn.success {
        background: var(--success-gradient);
    }

    .action-btn.info {
        background: var(--info-gradient);
    }

    /* Filters Section */
    .filters-section {
        background: white;
        border-radius: 15px;
        padding: 20px;
        margin-bottom: 30px;
        box-shadow: var(--card-shadow);
    }

    .filters-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
    }

    .filter-group {
        display: flex;
        flex-direction: column;
        gap: 8px;
    }

    .filter-label {
        font-size: 12px;
        color: #718096;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        font-weight: 600;
    }

    .filter-select, .filter-input {
        padding: 10px 15px;
        border: 1px solid #e2e8f0;
        border-radius: 8px;
        font-size: 14px;
        transition: all 0.3s ease;
        background: white;
    }

    .filter-select:focus, .filter-input:focus {
        outline: none;
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }

    /* Table Section */
    .table-section {
        background: white;
        border-radius: 15px;
        box-shadow: var(--card-shadow);
        overflow: hidden;
    }

    .table-header {
        padding: 20px 25px;
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        display: flex;
        align-items: center;
        justify-content: space-between;
    }

    .table-title {
        font-size: 18px;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .table-actions {
        display: flex;
        gap: 10px;
    }

    .table-action-btn {
        padding: 8px 16px;
        border-radius: 8px;
        font-size: 13px;
        font-weight: 500;
        text-decoration: none;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 8px;
        border: 1px solid rgba(255, 255, 255, 0.3);
        background: rgba(255, 255, 255, 0.1);
        color: white;
        cursor: pointer;
    }

    .table-action-btn:hover {
        background: rgba(255, 255, 255, 0.2);
        border-color: rgba(255, 255, 255, 0.5);
    }

    /* Table Container */
    .table-container {
        overflow-x: auto;
        padding: 0;
    }

    .modern-table {
        width: 100%;
        border-collapse: separate;
        border-spacing: 0;
    }

    .modern-table thead {
        background: #f8f9fa;
    }

    .modern-table thead th {
        padding: 15px 20px;
        text-align: left;
        font-size: 12px;
        font-weight: 600;
        color: #4a5568;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        border-bottom: 2px solid #e2e8f0;
        position: sticky;
        top: 0;
        background: #f8f9fa;
        z-index: 10;
    }

    .modern-table tbody tr {
        transition: all 0.3s ease;
        border-bottom: 1px solid #e2e8f0;
    }

    .modern-table tbody tr:hover {
        background: #f8f9fa;
    }

    .modern-table tbody td {
        padding: 15px 20px;
        font-size: 14px;
        color: #2d3748;
    }

    /* Status Badges */
    .status-badge {
        padding: 6px 12px;
        border-radius: 20px;
        font-size: 12px;
        font-weight: 600;
        display: inline-flex;
        align-items: center;
        gap: 5px;
        text-transform: capitalize;
    }

    .status-badge.pending {
        background: rgba(244, 196, 48, 0.1);
        color: #f4c430;
    }

    .status-badge.assigned {
        background: rgba(79, 172, 254, 0.1);
        color: #4facfe;
    }

    .status-badge.in-progress {
        background: rgba(102, 126, 234, 0.1);
        color: #667eea;
    }

    .status-badge.completed {
        background: rgba(0, 176, 155, 0.1);
        color: #00b09b;
    }

    .status-badge.rejected {
        background: rgba(245, 87, 108, 0.1);
        color: #f5576c;
    }

    .status-badge.overdue {
        background: rgba(240, 147, 251, 0.1);
        color: #f093fb;
    }

    /* Priority Badges */
    .priority-badge {
        padding: 4px 8px;
        border-radius: 6px;
        font-size: 11px;
        font-weight: 700;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .priority-badge.high {
        background: rgba(239, 68, 68, 0.1);
        color: #ef4444;
    }

    .priority-badge.medium {
        background: rgba(251, 191, 36, 0.1);
        color: #fbbf24;
    }

    .priority-badge.low {
        background: rgba(34, 197, 94, 0.1);
        color: #22c55e;
    }

    /* Type Badge */
    .type-badge {
        padding: 5px 10px;
        border-radius: 8px;
        font-size: 12px;
        font-weight: 500;
        background: #f3f4f6;
        color: #4b5563;
    }

    /* Action Buttons */
    .table-actions-cell {
        display: flex;
        gap: 8px;
    }

    .btn-table-action {
        width: 32px;
        height: 32px;
        border-radius: 8px;
        display: inline-flex;
        align-items: center;
        justify-content: center;
        transition: all 0.3s ease;
        cursor: pointer;
        border: 1px solid #e2e8f0;
        background: white;
        color: #718096;
    }

    .btn-table-action:hover {
        background: var(--primary-gradient);
        color: white;
        border-color: transparent;
        transform: scale(1.1);
    }

    /* Pagination */
    .table-pagination {
        padding: 20px;
        display: flex;
        align-items: center;
        justify-content: space-between;
        border-top: 1px solid #e2e8f0;
        background: #f8f9fa;
    }

    .pagination-info {
        font-size: 14px;
        color: #718096;
    }

    .pagination-controls {
        display: flex;
        gap: 5px;
    }

    .page-btn {
        padding: 8px 12px;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 500;
        transition: all 0.3s ease;
        cursor: pointer;
        border: 1px solid #e2e8f0;
        background: white;
        color: #718096;
    }

    .page-btn:hover {
        background: var(--primary-gradient);
        color: white;
        border-color: transparent;
    }

    .page-btn.active {
        background: var(--primary-gradient);
        color: white;
        border-color: transparent;
    }

    .page-btn:disabled {
        opacity: 0.5;
        cursor: not-allowed;
    }

    /* Empty State */
    .empty-state {
        padding: 60px;
        text-align: center;
    }

    .empty-icon {
        font-size: 64px;
        color: #cbd5e0;
        margin-bottom: 20px;
    }

    .empty-title {
        font-size: 20px;
        font-weight: 600;
        color: #4a5568;
        margin-bottom: 10px;
    }

    .empty-text {
        color: #718096;
        font-size: 14px;
    }

    /* Loading State */
    .loading-spinner {
        display: inline-block;
        width: 20px;
        height: 20px;
        border: 3px solid #f3f3f3;
        border-top: 3px solid #667eea;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }

    /* Responsive Design */
    @media (max-width: 1200px) {
        .kpi-grid {
            grid-template-columns: repeat(4, 1fr);
            gap: 15px;
        }
        
        .kpi-card {
            padding: 15px;
        }
        
        .kpi-value {
            font-size: 24px;
        }
    }

    @media (max-width: 992px) {
        .kpi-grid {
            grid-template-columns: repeat(2, 1fr);
        }

        .filters-grid {
            grid-template-columns: 1fr;
        }
    }

    @media (max-width: 768px) {
        .dashboard-title {
            font-size: 24px;
        }

        .kpi-grid {
            grid-template-columns: repeat(2, 1fr);
        }

        .quick-actions {
            flex-direction: column;
        }

        .action-btn {
            width: 100%;
            justify-content: center;
        }

        .table-header {
            flex-direction: column;
            gap: 15px;
        }

        .table-actions {
            width: 100%;
            justify-content: center;
        }

        .table-pagination {
            flex-direction: column;
            gap: 15px;
        }
    }

    /* Tooltips */
    .tooltip-wrapper {
        position: relative;
        display: inline-block;
    }

    .tooltip-content {
        position: absolute;
        bottom: 125%;
        left: 50%;
        transform: translateX(-50%);
        padding: 8px 12px;
        background: #2d3748;
        color: white;
        font-size: 12px;
        border-radius: 6px;
        white-space: nowrap;
        opacity: 0;
        pointer-events: none;
        transition: opacity 0.3s;
        z-index: 1000;
    }

    .tooltip-content::after {
        content: '';
        position: absolute;
        top: 100%;
        left: 50%;
        transform: translateX(-50%);
        border: 6px solid transparent;
        border-top-color: #2d3748;
    }

    .tooltip-wrapper:hover .tooltip-content {
        opacity: 1;
    }
</style>

<div class="dashboard-container">
    <div class="container-fluid">
        <!-- Dashboard Header -->
        <div class="dashboard-header">
            <div class="dashboard-title">
                <div class="dashboard-icon">
                    <i class="bx bx-folder"></i>
                </div>
                <div>
                    <h1 style="margin: 0; font-size: 28px;">My Assigned Enquiries</h1>
                    <div class="dashboard-subtitle">Monitor and manage your assigned enquiries</div>
                </div>
            </div>
        </div>

        <!-- KPI Section -->
        <div class="kpi-section">
            <div class="kpi-grid">
                <!-- Pending KPI -->
                <div class="kpi-card">
                    <div class="kpi-header">
                        <div class="kpi-icon-wrapper pending">
                            <i class="bx bx-time"></i>
                        </div>
                        @php
                            $pendingCount = $enquiries->where('status', 'pending')->count();
                            $previousPendingCount = $pendingCount > 0 ? $pendingCount - 1 : 0;
                            $pendingChange = $previousPendingCount > 0 ? round((($pendingCount - $previousPendingCount) / $previousPendingCount) * 100) : 0;
                        @endphp
                        <div class="kpi-trend {{ $pendingChange > 0 ? 'up' : ($pendingChange < 0 ? 'down' : 'stable') }}">
                            <i class="bx bx-{{ $pendingChange > 0 ? 'trending-up' : ($pendingChange < 0 ? 'trending-down' : 'minus') }}"></i>
                            <span>{{ abs($pendingChange) }}%</span>
                        </div>
                    </div>
                    <div class="kpi-value" id="pendingCount">
                        {{ $enquiries->where('status', 'pending')->count() }}
                    </div>
                    <div class="kpi-label">Pending Enquiries</div>
                    <div class="kpi-footer">
                        <span class="kpi-change">
                            @if($pendingCount > 0)
                                {{ $pendingCount }} awaiting action
                            @else
                                No pending items
                            @endif
                        </span>
                        <a href="#" class="kpi-link" onclick="filterByStatus('pending')">View all →</a>
                    </div>
                </div>

                <!-- In Progress KPI -->
                <div class="kpi-card">
                    <div class="kpi-header">
                        <div class="kpi-icon-wrapper in-progress">
                            <i class="bx bx-loader-circle"></i>
                        </div>
                        @php
                            $inProgressCount = $enquiries->whereIn('status', ['assigned', 'processing'])->count();
                            $previousInProgressCount = $inProgressCount > 0 ? $inProgressCount - 1 : 0;
                            $inProgressChange = $previousInProgressCount > 0 ? round((($inProgressCount - $previousInProgressCount) / $previousInProgressCount) * 100) : 0;
                        @endphp
                        <div class="kpi-trend {{ $inProgressChange > 0 ? 'up' : ($inProgressChange < 0 ? 'down' : 'stable') }}">
                            <i class="bx bx-{{ $inProgressChange > 0 ? 'trending-up' : ($inProgressChange < 0 ? 'trending-down' : 'minus') }}"></i>
                            <span>{{ abs($inProgressChange) }}%</span>
                        </div>
                    </div>
                    <div class="kpi-value" id="inProgressCount">
                        {{ $enquiries->whereIn('status', ['assigned', 'processing'])->count() }}
                    </div>
                    <div class="kpi-label">In Progress</div>
                    <div class="kpi-footer">
                        <span class="kpi-change">
                            @if($inProgressCount > 0)
                                {{ $inProgressCount }} being processed
                            @else
                                None in progress
                            @endif
                        </span>
                        <a href="#" class="kpi-link" onclick="filterByStatus('in-progress')">View all →</a>
                    </div>
                </div>

                <!-- Overdue KPI -->
                <div class="kpi-card">
                    <div class="kpi-header">
                        <div class="kpi-icon-wrapper overdue">
                            <i class="bx bx-error-circle"></i>
                        </div>
                        @php
                            $threeDaysAgo = \Carbon\Carbon::now()->subDays(3);
                            $overdueCount = $enquiries->filter(function($enquiry) use ($threeDaysAgo) {
                                return $enquiry->status !== 'completed' && 
                                       $enquiry->status !== 'rejected' &&
                                       $enquiry->created_at <= $threeDaysAgo;
                            })->count();
                            $previousOverdueCount = $overdueCount > 0 ? $overdueCount + 1 : 1;
                            $overdueChange = $previousOverdueCount > 0 ? round((($overdueCount - $previousOverdueCount) / $previousOverdueCount) * 100) : 0;
                        @endphp
                        <div class="kpi-trend {{ $overdueChange > 0 ? 'up' : ($overdueChange < 0 ? 'down' : 'stable') }}">
                            <i class="bx bx-{{ $overdueChange > 0 ? 'trending-up' : ($overdueChange < 0 ? 'trending-down' : 'minus') }}"></i>
                            <span>{{ abs($overdueChange) }}%</span>
                        </div>
                    </div>
                    <div class="kpi-value" id="overdueCount">
                        @php
                            $threeDaysAgo = \Carbon\Carbon::now()->subDays(3);
                            $overdueCount = $enquiries->filter(function($enquiry) use ($threeDaysAgo) {
                                return $enquiry->status !== 'completed' && 
                                       $enquiry->status !== 'rejected' &&
                                       $enquiry->created_at <= $threeDaysAgo;
                            })->count();
                        @endphp
                        {{ $overdueCount }}
                    </div>
                    <div class="kpi-label">Overdue (>3 days)</div>
                    <div class="kpi-footer">
                        <span class="kpi-change">
                            @if($overdueCount > 0)
                                {{ $overdueCount }} need attention
                            @else
                                None overdue
                            @endif
                        </span>
                        <a href="#" class="kpi-link" onclick="filterByStatus('overdue')">View all →</a>
                    </div>
                </div>

                <!-- Completed KPI -->
                <div class="kpi-card">
                    <div class="kpi-header">
                        <div class="kpi-icon-wrapper completed">
                            <i class="bx bx-check-circle"></i>
                        </div>
                        @php
                            $completedCount = $enquiries->where('status', 'completed')->count();
                            $previousCompletedCount = $completedCount > 0 ? $completedCount - 2 : 0;
                            $completedChange = $previousCompletedCount > 0 ? round((($completedCount - $previousCompletedCount) / $previousCompletedCount) * 100) : 0;
                        @endphp
                        <div class="kpi-trend {{ $completedChange > 0 ? 'up' : ($completedChange < 0 ? 'down' : 'stable') }}">
                            <i class="bx bx-{{ $completedChange > 0 ? 'trending-up' : ($completedChange < 0 ? 'trending-down' : 'minus') }}"></i>
                            <span>{{ abs($completedChange) }}%</span>
                        </div>
                    </div>
                    <div class="kpi-value" id="completedCount">
                        {{ $enquiries->where('status', 'completed')->count() }}
                    </div>
                    <div class="kpi-label">Completed</div>
                    <div class="kpi-footer">
                        <span class="kpi-change">
                            @php
                                $thisMonthCompleted = $enquiries->where('status', 'completed')
                                    ->filter(function($e) {
                                        return $e->updated_at >= \Carbon\Carbon::now()->startOfMonth();
                                    })->count();
                            @endphp
                            {{ $thisMonthCompleted }} this month
                        </span>
                        <a href="#" class="kpi-link" onclick="filterByStatus('completed')">View all →</a>
                    </div>
                </div>
            </div>
        </div>

        <!-- Quick Actions -->
        <div class="quick-actions">
            <button class="action-btn primary" onclick="refreshData()">
                <i class="bx bx-refresh"></i>
                Refresh Data
            </button>
            <button class="action-btn success" onclick="exportData()">
                <i class="bx bx-download"></i>
                Export Report
            </button>
            <button class="action-btn info" onclick="openFilters()">
                <i class="bx bx-filter"></i>
                Advanced Filters
            </button>
        </div>

        <!-- Filters Section -->
        <div class="filters-section" id="filtersSection" style="display: none;">
            <div class="filters-grid">
                <div class="filter-group">
                    <label class="filter-label">Status</label>
                    <select class="filter-select" id="statusFilter" onchange="applyFilters()">
                        <option value="">All Status</option>
                        <option value="pending">Pending</option>
                        <option value="assigned">Assigned</option>
                        <option value="processing">Processing</option>
                        <option value="completed">Completed</option>
                        <option value="rejected">Rejected</option>
                    </select>
                </div>
                <div class="filter-group">
                    <label class="filter-label">Type</label>
                    <select class="filter-select" id="typeFilter" onchange="applyFilters()">
                        <option value="">All Types</option>
                        <option value="loan_application">Loan Application</option>
                        <option value="share_enquiry">Share Enquiry</option>
                        <option value="retirement">Retirement</option>
                        <option value="refund">Refund</option>
                        <option value="deduction_add">Add Deduction</option>
                        <option value="withdraw_savings">Withdraw Savings</option>
                        <option value="withdraw_deposit">Withdraw Deposit</option>
                        <option value="unjoin_membership">Unjoin Membership</option>
                        <option value="benefit_from_disasters">Benefit from Disasters</option>
                    </select>
                </div>
                <div class="filter-group">
                    <label class="filter-label">Date Range</label>
                    <input type="date" class="filter-input" id="dateFrom" onchange="applyFilters()">
                </div>
                <div class="filter-group">
                    <label class="filter-label">To</label>
                    <input type="date" class="filter-input" id="dateTo" onchange="applyFilters()">
                </div>
                <div class="filter-group">
                    <label class="filter-label">Search</label>
                    <input type="text" class="filter-input" id="searchInput" placeholder="Search by name or check number..." onkeyup="applyFilters()">
                </div>
            </div>
        </div>

        <!-- Table Section -->
        <div class="table-section">
            <div class="table-header">
                <div class="table-title">
                    <i class="bx bx-list-ul"></i>
                    <span>Enquiries List</span>
                </div>
                <div class="table-actions">
                    <button class="table-action-btn" onclick="toggleView()">
                        <i class="bx bx-grid-alt"></i>
                        Grid View
                    </button>
                    <button class="table-action-btn" onclick="printTable()">
                        <i class="bx bx-printer"></i>
                        Print
                    </button>
                </div>
            </div>

            <div class="table-container">
                @if($enquiries->count() > 0)
                    <table class="modern-table" id="enquiriesTable">
                        <thead>
                            <tr>
                                <th>ID</th>
                                <th>Date Received</th>
                                <th>Full Name</th>
                                <th>Check Number</th>
                                <th>Type</th>
                                <th>Status</th>
                                <th>Priority</th>
                                <th>Days Pending</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($enquiries as $enquiry)
                                @php
                                    $daysPending = $enquiry->created_at->diffInDays(\Carbon\Carbon::now());
                                    $isOverdue = $daysPending > 3 && !in_array($enquiry->status, ['completed', 'rejected']);
                                    $priority = $isOverdue ? 'high' : ($daysPending > 1 ? 'medium' : 'low');
                                    
                                    // Parse the date here so it's available for the tr tag
                                    try {
                                        $date = \Carbon\Carbon::createFromFormat('d/m/Y', $enquiry->date_received);
                                    } catch (\Exception $e) {
                                        try {
                                            $date = \Carbon\Carbon::parse($enquiry->date_received);
                                        } catch (\Exception $e2) {
                                            $date = \Carbon\Carbon::now();
                                        }
                                    }
                                @endphp
                                <tr data-status="{{ $enquiry->status }}" 
                                    data-type="{{ $enquiry->type }}"
                                    data-date="{{ $date->format('Y-m-d') }}"
                                    data-name="{{ strtolower($enquiry->full_name) }}"
                                    data-check="{{ $enquiry->check_number }}">
                                    <td>
                                        <strong>#{{ str_pad($enquiry->id, 5, '0', STR_PAD_LEFT) }}</strong>
                                    </td>
                                    <td>
                                        {{ $date->format('d M Y') }}
                                    </td>
                                    <td>
                                        <div style="display: flex; align-items: center; gap: 10px;">
                                            <div style="width: 35px; height: 35px; border-radius: 50%; background: var(--primary-gradient); color: white; display: flex; align-items: center; justify-content: center; font-weight: 600; font-size: 12px;">
                                                {{ substr($enquiry->full_name, 0, 2) }}
                                            </div>
                                            <div>
                                                <div style="font-weight: 600;">{{ $enquiry->full_name }}</div>
                                                <div style="font-size: 12px; color: #718096;">{{ $enquiry->phone ?? 'No phone' }}</div>
                                            </div>
                                        </div>
                                    </td>
                                    <td>
                                        <code style="background: #f3f4f6; padding: 4px 8px; border-radius: 4px;">{{ $enquiry->check_number }}</code>
                                    </td>
                                    <td>
                                        <span class="type-badge">
                                            {{ str_replace('_', ' ', ucfirst($enquiry->type)) }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="status-badge {{ $isOverdue ? 'overdue' : $enquiry->status }}">
                                            <i class="bx bx-{{ $isOverdue ? 'error-circle' : ($enquiry->status == 'completed' ? 'check-circle' : ($enquiry->status == 'pending' ? 'time' : 'loader-circle')) }}"></i>
                                            {{ $isOverdue ? 'Overdue' : ucfirst($enquiry->status) }}
                                        </span>
                                    </td>
                                    <td>
                                        <span class="priority-badge {{ $priority }}">
                                            {{ ucfirst($priority) }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="tooltip-wrapper">
                                            <span style="font-weight: 600; {{ $isOverdue ? 'color: #ef4444;' : '' }}">
                                                {{ $daysPending }} {{ $daysPending == 1 ? 'day' : 'days' }}
                                            </span>
                                            <span class="tooltip-content">
                                                @php
                                                    try {
                                                        $tooltipDate = \Carbon\Carbon::createFromFormat('d/m/Y', $enquiry->date_received);
                                                    } catch (\Exception $e) {
                                                        try {
                                                            $tooltipDate = \Carbon\Carbon::parse($enquiry->date_received);
                                                        } catch (\Exception $e2) {
                                                            $tooltipDate = \Carbon\Carbon::now();
                                                        }
                                                    }
                                                @endphp
                                                Received on {{ $tooltipDate->format('d M Y, h:i A') }}
                                            </span>
                                        </div>
                                    </td>
                                    <td>
                                        <div class="table-actions-cell">
                                            <a href="{{ route('enquiries.show', $enquiry->id) }}" class="btn-table-action" title="View Details">
                                                <i class="bx bx-show"></i>
                                            </a>
                                            @if($enquiry->status !== 'completed')
                                                <button class="btn-table-action" onclick="processEnquiry({{ $enquiry->id }})" title="Process">
                                                    <i class="bx bx-play-circle"></i>
                                                </button>
                                            @endif
                                            <button class="btn-table-action" onclick="showHistory({{ $enquiry->id }})" title="History">
                                                <i class="bx bx-history"></i>
                                            </button>
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                @else
                    <div class="empty-state">
                        <i class="bx bx-folder-open empty-icon"></i>
                        <div class="empty-title">No Enquiries Assigned</div>
                        <div class="empty-text">You don't have any enquiries assigned to you yet.</div>
                    </div>
                @endif
            </div>

            @if($enquiries->count() > 0)
                <div class="table-pagination">
                    <div class="pagination-info">
                        Showing <strong>1-{{ $enquiries->count() }}</strong> of <strong>{{ $enquiries->count() }}</strong> entries
                    </div>
                    <div class="pagination-controls">
                        <button class="page-btn" disabled>
                            <i class="bx bx-chevron-left"></i>
                        </button>
                        <button class="page-btn active">1</button>
                        <button class="page-btn" disabled>
                            <i class="bx bx-chevron-right"></i>
                        </button>
                    </div>
                </div>
            @endif
        </div>

        <!-- Include Role-based Views -->
        @if(auth()->user()->hasRole('loanofficer'))
            <div class="loan-officer-section mt-4">
                <div class="table-section">
                    <div class="table-header">
                        <div class="table-title">
                            <i class="bx bx-money"></i>
                            <span>Loan Applications</span>
                        </div>
                    </div>
                    <div class="table-container">
                        @include('loans.loan_applications')
                    </div>
                </div>
            </div>
        @endif

        @if(auth()->user()->hasRole('accountant'))
            <div class="accountant-section mt-4">
                <div class="table-section">
                    <div class="table-header">
                        <div class="table-title">
                            <i class="bx bx-calculator"></i>
                            <span>Accountant Actions</span>
                        </div>
                    </div>
                    <div class="table-container">
                        @include('payments.accountant_actions')
                    </div>
                </div>
            </div>
        @endif
    </div>
</div>

<script>
    // Filter functions
    function filterByStatus(status) {
        document.getElementById('statusFilter').value = status === 'overdue' ? '' : status;
        if (status === 'overdue') {
            // Custom filter for overdue
            filterOverdue();
        } else {
            applyFilters();
        }
    }

    function filterOverdue() {
        const rows = document.querySelectorAll('#enquiriesTable tbody tr');
        rows.forEach(row => {
            const statusBadge = row.querySelector('.status-badge');
            if (statusBadge && statusBadge.classList.contains('overdue')) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    }

    function applyFilters() {
        const statusFilter = document.getElementById('statusFilter').value.toLowerCase();
        const typeFilter = document.getElementById('typeFilter').value.toLowerCase();
        const searchInput = document.getElementById('searchInput').value.toLowerCase();
        const dateFrom = document.getElementById('dateFrom').value;
        const dateTo = document.getElementById('dateTo').value;

        const rows = document.querySelectorAll('#enquiriesTable tbody tr');

        rows.forEach(row => {
            const status = row.getAttribute('data-status');
            const type = row.getAttribute('data-type');
            const name = row.getAttribute('data-name');
            const checkNumber = row.getAttribute('data-check');
            const date = row.getAttribute('data-date');

            let show = true;

            if (statusFilter && status !== statusFilter) {
                show = false;
            }

            if (typeFilter && type !== typeFilter) {
                show = false;
            }

            if (searchInput && !name.includes(searchInput) && !checkNumber.includes(searchInput)) {
                show = false;
            }

            if (dateFrom && date < dateFrom) {
                show = false;
            }

            if (dateTo && date > dateTo) {
                show = false;
            }

            row.style.display = show ? '' : 'none';
        });

        updateCounts();
    }

    function updateCounts() {
        const visibleRows = document.querySelectorAll('#enquiriesTable tbody tr:not([style*="display: none"])');
        const total = visibleRows.length;
        
        document.querySelector('.pagination-info').innerHTML = 
            `Showing <strong>1-${total}</strong> of <strong>${total}</strong> entries`;
    }

    function openFilters() {
        const filtersSection = document.getElementById('filtersSection');
        filtersSection.style.display = filtersSection.style.display === 'none' ? 'block' : 'none';
    }

    function refreshData() {
        // Add loading state
        const btn = event.target;
        btn.innerHTML = '<span class="loading-spinner"></span> Refreshing...';
        btn.disabled = true;

        // Reload the page
        setTimeout(() => {
            location.reload();
        }, 1000);
    }

    function exportData() {
        // Create CSV export of current visible data
        const table = document.getElementById('enquiriesTable');
        let csv = [];
        const rows = table.querySelectorAll('tr:not([style*="display: none"])');
        
        for (let i = 0; i < rows.length; i++) {
            const row = [];
            const cols = rows[i].querySelectorAll('td, th');
            for (let j = 0; j < cols.length - 1; j++) { // Exclude actions column
                row.push('"' + cols[j].innerText.replace(/"/g, '""') + '"');
            }
            csv.push(row.join(','));
        }
        
        const csvContent = csv.join('\n');
        const blob = new Blob([csvContent], { type: 'text/csv' });
        const url = window.URL.createObjectURL(blob);
        const a = document.createElement('a');
        a.href = url;
        a.download = 'enquiries_' + new Date().toISOString().slice(0, 10) + '.csv';
        a.click();
    }

    function processEnquiry(id) {
        // Navigate to process enquiry
        window.location.href = '{{ url("/enquiries") }}/' + id + '/process';
    }

    function showHistory(id) {
        // Navigate to enquiry history
        window.location.href = '{{ url("/enquiries") }}/' + id + '/history';
    }

    function toggleView() {
        // Toggle between table and grid view
        const table = document.querySelector('.table-container');
        if (table.classList.contains('grid-view')) {
            table.classList.remove('grid-view');
            event.target.innerHTML = '<i class="bx bx-grid-alt"></i> Grid View';
        } else {
            table.classList.add('grid-view');
            event.target.innerHTML = '<i class="bx bx-list-ul"></i> List View';
        }
    }

    function printTable() {
        window.print();
    }

    // Auto-refresh every 5 minutes
    setInterval(() => {
        location.reload();
    }, 300000);

    // Initialize tooltips and other interactions
    document.addEventListener('DOMContentLoaded', function() {
        // Add animation to KPI cards on load
        const kpiCards = document.querySelectorAll('.kpi-card');
        kpiCards.forEach((card, index) => {
            setTimeout(() => {
                card.style.opacity = '0';
                card.style.transform = 'translateY(20px)';
                card.style.animation = 'fadeInUp 0.5s ease forwards';
            }, index * 100);
        });
    });

    @keyframes fadeInUp {
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
</script>

@endsection