@extends('layouts.app')

@section('content')
<style>
    /* Modern File Detail View Styles */
    :root {
        --primary-gradient: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        --secondary-gradient: linear-gradient(135deg, #f093fb 0%, #f5576c 100%);
        --success-gradient: linear-gradient(135deg, #00b09b 0%, #96c93d 100%);
        --info-gradient: linear-gradient(135deg, #4facfe 0%, #00f2fe 100%);
        --card-shadow: 0 10px 30px rgba(0, 0, 0, 0.1);
        --card-hover-shadow: 0 20px 40px rgba(0, 0, 0, 0.15);
    }

    /* Main Container Styling */
    .file-detail-container {
        padding: 30px 0;
        background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
        min-height: calc(100vh - 60px);
    }

    /* Header Section */
    .file-header {
        background: white;
        border-radius: 15px;
        padding: 30px;
        margin-bottom: 30px;
        box-shadow: var(--card-shadow);
        position: relative;
        overflow: hidden;
    }

    .file-header::before {
        content: '';
        position: absolute;
        top: 0;
        left: 0;
        right: 0;
        height: 5px;
        background: var(--primary-gradient);
    }

    .file-title {
        font-size: 28px;
        font-weight: 700;
        color: #2d3748;
        margin-bottom: 15px;
        display: flex;
        align-items: center;
        gap: 15px;
    }

    .file-icon {
        width: 50px;
        height: 50px;
        background: var(--primary-gradient);
        border-radius: 12px;
        display: flex;
        align-items: center;
        justify-content: center;
        color: white;
        font-size: 24px;
        animation: pulse 2s infinite;
    }

    @keyframes pulse {
        0%, 100% { transform: scale(1); }
        50% { transform: scale(1.05); }
    }

    .file-meta {
        display: flex;
        gap: 30px;
        flex-wrap: wrap;
        margin-top: 20px;
    }

    .meta-item {
        display: flex;
        flex-direction: column;
        gap: 5px;
    }

    .meta-label {
        font-size: 12px;
        color: #718096;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }

    .meta-value {
        font-size: 16px;
        font-weight: 600;
        color: #2d3748;
    }

    /* Stats Cards */
    .stats-grid {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
        gap: 20px;
        margin-bottom: 30px;
    }

    .stat-card {
        background: white;
        border-radius: 15px;
        padding: 20px;
        box-shadow: var(--card-shadow);
        transition: all 0.3s ease;
        position: relative;
        overflow: hidden;
    }

    .stat-card:hover {
        transform: translateY(-5px);
        box-shadow: var(--card-hover-shadow);
    }

    .stat-card::after {
        content: '';
        position: absolute;
        top: 0;
        right: 0;
        width: 100px;
        height: 100px;
        background: var(--primary-gradient);
        opacity: 0.05;
        border-radius: 50%;
        transform: translate(30px, -30px);
    }

    .stat-icon {
        width: 40px;
        height: 40px;
        border-radius: 10px;
        display: flex;
        align-items: center;
        justify-content: center;
        font-size: 20px;
        color: white;
        margin-bottom: 15px;
    }

    .stat-icon.folios { background: var(--info-gradient); }
    .stat-icon.status { background: var(--success-gradient); }
    .stat-icon.department { background: var(--secondary-gradient); }
    .stat-icon.updated { background: var(--primary-gradient); }

    .stat-value {
        font-size: 28px;
        font-weight: 700;
        color: #2d3748;
        margin-bottom: 5px;
    }

    .stat-label {
        font-size: 14px;
        color: #718096;
    }

    /* Main Content Grid */
    .content-grid {
        display: grid;
        grid-template-columns: 350px 1fr;
        gap: 30px;
        margin-top: 30px;
    }

    /* Folio List Sidebar */
    .folio-sidebar {
        background: white;
        border-radius: 15px;
        box-shadow: var(--card-shadow);
        overflow: hidden;
        max-height: 80vh;
        display: flex;
        flex-direction: column;
    }

    .sidebar-header {
        background: var(--primary-gradient);
        color: white;
        padding: 20px;
        display: flex;
        align-items: center;
        justify-content: space-between;
    }

    .sidebar-title {
        font-size: 18px;
        font-weight: 600;
        display: flex;
        align-items: center;
        gap: 10px;
    }

    .folio-count {
        background: rgba(255, 255, 255, 0.2);
        padding: 4px 12px;
        border-radius: 20px;
        font-size: 14px;
    }

    /* Search Box */
    .folio-search {
        padding: 15px;
        background: #f8f9fa;
        border-bottom: 1px solid #e2e8f0;
    }

    .search-input {
        width: 100%;
        padding: 10px 15px;
        border: 1px solid #e2e8f0;
        border-radius: 10px;
        font-size: 14px;
        transition: all 0.3s ease;
    }

    .search-input:focus {
        outline: none;
        border-color: #667eea;
        box-shadow: 0 0 0 3px rgba(102, 126, 234, 0.1);
    }

    /* Folio List */
    .folio-list {
        flex: 1;
        overflow-y: auto;
        padding: 10px;
    }

    .folio-list::-webkit-scrollbar {
        width: 6px;
    }

    .folio-list::-webkit-scrollbar-track {
        background: #f1f1f1;
        border-radius: 3px;
    }

    .folio-list::-webkit-scrollbar-thumb {
        background: var(--primary-gradient);
        border-radius: 3px;
    }

    .folio-item {
        background: #f8f9fa;
        border-radius: 10px;
        padding: 15px;
        margin-bottom: 10px;
        cursor: pointer;
        transition: all 0.3s ease;
        border: 2px solid transparent;
        position: relative;
    }

    .folio-item:hover {
        background: white;
        box-shadow: 0 5px 15px rgba(0, 0, 0, 0.08);
        transform: translateX(5px);
    }

    .folio-item.active {
        background: white;
        border-color: #667eea;
        box-shadow: 0 5px 20px rgba(102, 126, 234, 0.2);
    }

    .folio-item.active::before {
        content: '';
        position: absolute;
        left: 0;
        top: 0;
        bottom: 0;
        width: 4px;
        background: var(--primary-gradient);
        border-radius: 0 4px 4px 0;
    }

    .folio-number {
        font-size: 12px;
        font-weight: 600;
        color: #667eea;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 8px;
    }

    .folio-name {
        font-size: 14px;
        font-weight: 500;
        color: #2d3748;
        margin-bottom: 5px;
        white-space: nowrap;
        overflow: hidden;
        text-overflow: ellipsis;
    }

    .folio-date {
        font-size: 12px;
        color: #718096;
        display: flex;
        align-items: center;
        gap: 5px;
    }

    /* Detail View */
    .detail-container {
        background: white;
        border-radius: 15px;
        box-shadow: var(--card-shadow);
        overflow: hidden;
        min-height: 80vh;
        display: flex;
        flex-direction: column;
    }

    .detail-header {
        background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
        color: white;
        padding: 25px;
        border-bottom: 1px solid #e2e8f0;
    }

    .detail-title {
        font-size: 20px;
        font-weight: 600;
        margin-bottom: 10px;
    }

    .detail-subtitle {
        font-size: 14px;
        opacity: 0.9;
    }

    /* Empty State */
    .empty-state {
        flex: 1;
        display: flex;
        flex-direction: column;
        align-items: center;
        justify-content: center;
        padding: 40px;
        text-align: center;
    }

    .empty-icon {
        font-size: 64px;
        color: #cbd5e0;
        margin-bottom: 20px;
    }

    .empty-title {
        font-size: 20px;
        font-weight: 600;
        color: #4a5568;
        margin-bottom: 10px;
    }

    .empty-text {
        color: #718096;
        font-size: 14px;
    }

    /* Detail Content */
    .detail-content {
        padding: 25px;
        flex: 1;
        overflow-y: auto;
    }

    .info-grid {
        display: grid;
        grid-template-columns: repeat(2, 1fr);
        gap: 20px;
        margin-bottom: 30px;
    }

    .info-item {
        padding: 15px;
        background: #f8f9fa;
        border-radius: 10px;
        border-left: 4px solid #667eea;
    }

    .info-label {
        font-size: 12px;
        color: #718096;
        text-transform: uppercase;
        letter-spacing: 0.5px;
        margin-bottom: 5px;
    }

    .info-value {
        font-size: 16px;
        font-weight: 600;
        color: #2d3748;
    }

    /* PDF Viewer Section */
    .pdf-section {
        margin-top: 30px;
    }

    .pdf-header {
        display: flex;
        align-items: center;
        justify-content: space-between;
        margin-bottom: 20px;
        padding-bottom: 15px;
        border-bottom: 2px solid #e2e8f0;
    }

    .pdf-title {
        font-size: 18px;
        font-weight: 600;
        color: #2d3748;
    }

    .pdf-actions {
        display: flex;
        gap: 10px;
    }

    .btn-action {
        padding: 8px 16px;
        border-radius: 8px;
        font-size: 14px;
        font-weight: 500;
        text-decoration: none;
        transition: all 0.3s ease;
        display: inline-flex;
        align-items: center;
        gap: 8px;
        border: none;
        cursor: pointer;
    }

    .btn-download {
        background: var(--success-gradient);
        color: white;
    }

    .btn-download:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 20px rgba(0, 176, 155, 0.3);
    }

    .btn-print {
        background: #f8f9fa;
        color: #4a5568;
        border: 1px solid #e2e8f0;
    }

    .btn-print:hover {
        background: #e2e8f0;
    }

    .btn-fullscreen {
        background: var(--info-gradient);
        color: white;
    }

    .btn-fullscreen:hover {
        transform: translateY(-2px);
        box-shadow: 0 5px 20px rgba(79, 172, 254, 0.3);
    }

    /* PDF Container */
    .pdf-container {
        background: #f8f9fa;
        border-radius: 10px;
        padding: 20px;
        min-height: 600px;
        position: relative;
    }

    .pdf-viewer {
        width: 100%;
        height: 600px;
        border-radius: 8px;
        box-shadow: 0 5px 20px rgba(0, 0, 0, 0.1);
    }

    /* Responsive Design */
    @media (max-width: 1024px) {
        .content-grid {
            grid-template-columns: 1fr;
        }

        .folio-sidebar {
            max-height: 400px;
        }
    }

    @media (max-width: 768px) {
        .file-meta {
            flex-direction: column;
            gap: 15px;
        }

        .stats-grid {
            grid-template-columns: 1fr 1fr;
        }

        .info-grid {
            grid-template-columns: 1fr;
        }

        .pdf-actions {
            flex-direction: column;
        }

        .btn-action {
            width: 100%;
            justify-content: center;
        }
    }

    /* Loading Animation */
    .loading {
        display: inline-block;
        width: 20px;
        height: 20px;
        border: 3px solid #f3f3f3;
        border-top: 3px solid #667eea;
        border-radius: 50%;
        animation: spin 1s linear infinite;
    }

    @keyframes spin {
        0% { transform: rotate(0deg); }
        100% { transform: rotate(360deg); }
    }

    /* Fade In Animation */
    .fade-in {
        animation: fadeIn 0.5s ease-in;
    }

    @keyframes fadeIn {
        from { opacity: 0; transform: translateY(10px); }
        to { opacity: 1; transform: translateY(0); }
    }
</style>

<div class="file-detail-container">
    <div class="container">
        <!-- Header Section -->
        <div class="file-header fade-in">
            <div class="file-title">
                <div class="file-icon">
                    <i class="bx bx-folder"></i>
                </div>
                <span>{{ $file->file_subject ?? 'File Details' }}</span>
            </div>
            
            <div class="file-meta">
                <div class="meta-item">
                    <span class="meta-label">Reference Number</span>
                    <span class="meta-value">{{ $file->reference_number ?? 'N/A' }}</span>
                </div>
                <div class="meta-item">
                    <span class="meta-label">Department</span>
                    <span class="meta-value">{{ $file->department->name ?? 'N/A' }}</span>
                </div>
                <div class="meta-item">
                    <span class="meta-label">File Series</span>
                    <span class="meta-value">{{ $file->fileSeries->name ?? 'N/A' }}</span>
                </div>
                <div class="meta-item">
                    <span class="meta-label">Created Date</span>
                    <span class="meta-value">{{ $file->created_at ? $file->created_at->format('d M Y') : 'N/A' }}</span>
                </div>
            </div>
        </div>

        <!-- Statistics Cards -->
        <div class="stats-grid fade-in">
            <div class="stat-card">
                <div class="stat-icon folios">
                    <i class="bx bx-file"></i>
                </div>
                <div class="stat-value">{{ $file->folios->count() }}</div>
                <div class="stat-label">Total Folios</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon status">
                    <i class="bx bx-check-circle"></i>
                </div>
                <div class="stat-value">{{ $file->is_active ? 'Active' : 'Inactive' }}</div>
                <div class="stat-label">Status</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon department">
                    <i class="bx bx-building"></i>
                </div>
                <div class="stat-value">{{ $file->branch->name ?? 'Main' }}</div>
                <div class="stat-label">Branch</div>
            </div>
            
            <div class="stat-card">
                <div class="stat-icon updated">
                    <i class="bx bx-calendar"></i>
                </div>
                <div class="stat-value">{{ $file->updated_at ? $file->updated_at->diffForHumans() : 'Never' }}</div>
                <div class="stat-label">Last Updated</div>
            </div>
        </div>

        <!-- Main Content Grid -->
        <div class="content-grid">
            <!-- Folio List Sidebar -->
            <div class="folio-sidebar fade-in">
                <div class="sidebar-header">
                    <div class="sidebar-title">
                        <i class="bx bx-folder-open"></i>
                        <span>Folios</span>
                    </div>
                    <div class="folio-count">{{ $file->folios->count() }}</div>
                </div>
                
                <!-- Search Box -->
                <div class="folio-search">
                    <input type="text" 
                           class="search-input" 
                           id="folioSearch" 
                           placeholder="Search folios...">
                </div>
                
                <!-- Folio List -->
                <div class="folio-list" id="folioList">
                    @php $totalFolios = $file->folios->count(); @endphp
                    @forelse ($file->folios->sortByDesc('created_at') as $index => $folio)
                        <div class="folio-item {{ $index === 0 ? 'active' : '' }}" 
                             data-folio-id="{{ $folio->id }}"
                             onclick="showFolioDetails('{{ $folio->id }}')">
                            <div class="folio-number">Folio {{ $totalFolios - $loop->index }}</div>
                            <div class="folio-name">{{ basename($folio->file_path) }}</div>
                            <div class="folio-date">
                                <i class="bx bx-time-five"></i>
                                {{ $folio->created_at ? $folio->created_at->format('d M Y, h:i A') : 'N/A' }}
                            </div>
                        </div>
                    @empty
                        <div class="empty-state">
                            <i class="bx bx-folder-open empty-icon"></i>
                            <div class="empty-text">No folios available</div>
                        </div>
                    @endforelse
                </div>
            </div>

            <!-- Detail View -->
            <div class="detail-container fade-in">
                @if($file->folios->count() > 0)
                    @foreach ($file->folios->sortByDesc('created_at') as $index => $folio)
                        <div class="folio-detail" 
                             id="folio-detail-{{ $folio->id }}" 
                             style="display: {{ $index === 0 ? 'block' : 'none' }};">
                            
                            <div class="detail-header">
                                <div class="detail-title">Folio Details</div>
                                <div class="detail-subtitle">{{ basename($folio->file_path) }}</div>
                            </div>
                            
                            <div class="detail-content">
                                <!-- Info Grid -->
                                <div class="info-grid">
                                    <div class="info-item">
                                        <div class="info-label">Enquirer Name</div>
                                        <div class="info-value">
                                            {{ $folio->folioable ? $folio->folioable->full_name : 'Not Available' }}
                                        </div>
                                    </div>
                                    
                                    <div class="info-item">
                                        <div class="info-label">Check Number</div>
                                        <div class="info-value">
                                            {{ $folio->folioable ? $folio->folioable->check_number : 'Not Available' }}
                                        </div>
                                    </div>
                                    
                                    <div class="info-item">
                                        <div class="info-label">File Type</div>
                                        <div class="info-value">
                                            {{ strtoupper(pathinfo($folio->file_path, PATHINFO_EXTENSION)) }}
                                        </div>
                                    </div>
                                    
                                    <div class="info-item">
                                        <div class="info-label">Upload Date</div>
                                        <div class="info-value">
                                            {{ $folio->created_at ? $folio->created_at->format('d M Y, h:i A') : 'N/A' }}
                                        </div>
                                    </div>
                                </div>

                                <!-- PDF Section -->
                                @if (strtolower(pathinfo($folio->file_path, PATHINFO_EXTENSION)) === 'pdf')
                                    <div class="pdf-section">
                                        <div class="pdf-header">
                                            <div class="pdf-title">
                                                <i class="bx bx-file-pdf"></i> Document Preview
                                            </div>
                                            <div class="pdf-actions">
                                                <button class="btn-action btn-print" onclick="printPDF('{{ $folio->id }}')">
                                                    <i class="bx bx-printer"></i> Print
                                                </button>
                                                <a href="{{ asset('/' . $folio->file_path) }}" 
                                                   download 
                                                   class="btn-action btn-download">
                                                    <i class="bx bx-download"></i> Download
                                                </a>
                                                <button class="btn-action btn-fullscreen" onclick="fullscreenPDF('{{ $folio->id }}')">
                                                    <i class="bx bx-fullscreen"></i> Fullscreen
                                                </button>
                                            </div>
                                        </div>
                                        
                                        <div class="pdf-container">
                                            <embed src="{{ asset('/' . $folio->file_path) }}" 
                                                   type="application/pdf" 
                                                   class="pdf-viewer"
                                                   id="pdf-viewer-{{ $folio->id }}">
                                        </div>
                                    </div>
                                @else
                                    <div class="pdf-section">
                                        <div class="pdf-header">
                                            <div class="pdf-title">
                                                <i class="bx bx-file"></i> File Actions
                                            </div>
                                        </div>
                                        <div class="pdf-container" style="text-align: center; padding: 60px;">
                                            <i class="bx bx-file" style="font-size: 64px; color: #cbd5e0; margin-bottom: 20px;"></i>
                                            <p style="color: #718096; margin-bottom: 20px;">This file type cannot be previewed</p>
                                            <a href="{{ asset('/' . $folio->file_path) }}" 
                                               download 
                                               class="btn-action btn-download">
                                                <i class="bx bx-download"></i> Download File
                                            </a>
                                        </div>
                                    </div>
                                @endif
                            </div>
                        </div>
                    @endforeach
                @else
                    <div class="empty-state">
                        <i class="bx bx-folder-open empty-icon"></i>
                        <div class="empty-title">No Folio Selected</div>
                        <div class="empty-text">Select a folio from the list to view details</div>
                    </div>
                @endif
            </div>
        </div>
    </div>
</div>

<script>
    // Show folio details
    function showFolioDetails(folioId) {
        // Hide all folio details
        document.querySelectorAll('.folio-detail').forEach(detail => {
            detail.style.display = 'none';
        });
        
        // Show selected folio detail
        const selectedDetail = document.getElementById('folio-detail-' + folioId);
        if (selectedDetail) {
            selectedDetail.style.display = 'block';
            selectedDetail.classList.add('fade-in');
        }
        
        // Update active state in list
        document.querySelectorAll('.folio-item').forEach(item => {
            item.classList.remove('active');
        });
        
        const selectedItem = document.querySelector(`[data-folio-id="${folioId}"]`);
        if (selectedItem) {
            selectedItem.classList.add('active');
        }
    }

    // Search functionality
    document.getElementById('folioSearch')?.addEventListener('input', function(e) {
        const searchTerm = e.target.value.toLowerCase();
        const folioItems = document.querySelectorAll('.folio-item');
        
        folioItems.forEach(item => {
            const folioName = item.querySelector('.folio-name')?.textContent.toLowerCase();
            const folioNumber = item.querySelector('.folio-number')?.textContent.toLowerCase();
            
            if (folioName?.includes(searchTerm) || folioNumber?.includes(searchTerm)) {
                item.style.display = 'block';
            } else {
                item.style.display = 'none';
            }
        });
    });

    // Print PDF
    function printPDF(folioId) {
        const pdfViewer = document.getElementById('pdf-viewer-' + folioId);
        if (pdfViewer) {
            pdfViewer.contentWindow.print();
        }
    }

    // Fullscreen PDF
    function fullscreenPDF(folioId) {
        const pdfViewer = document.getElementById('pdf-viewer-' + folioId);
        if (pdfViewer) {
            if (pdfViewer.requestFullscreen) {
                pdfViewer.requestFullscreen();
            } else if (pdfViewer.webkitRequestFullscreen) {
                pdfViewer.webkitRequestFullscreen();
            } else if (pdfViewer.msRequestFullscreen) {
                pdfViewer.msRequestFullscreen();
            }
        }
    }

    // Initialize first folio as active on page load
    document.addEventListener('DOMContentLoaded', function() {
        const firstFolio = document.querySelector('.folio-item');
        if (firstFolio) {
            const folioId = firstFolio.getAttribute('data-folio-id');
            if (folioId) {
                showFolioDetails(folioId);
            }
        }
    });
</script>

@endsection