# Production Deployment Guide - Loan System Improvements

## Overview
This guide provides step-by-step instructions for deploying the normalized loan system improvements to production while maintaining 100% backward compatibility.

## Pre-Deployment Checklist

### 1. Backup Current State
```bash
# Backup database
mysqldump -u root -p uraerp > uraerp_backup_$(date +%Y%m%d_%H%M%S).sql

# Backup application files
tar -czf uraerp_app_backup_$(date +%Y%m%d_%H%M%S).tar.gz /path/to/uraerp/
```

### 2. Verify Migration Files
Ensure these migration files are present:
- `2025_08_31_120000_create_banks_table.php`
- `2025_08_31_224531_create_loan_offer_approvals_table.php`
- `2025_08_31_224719_create_loan_disbursements_table.php`
- `2025_08_31_224758_create_loan_offer_topups_table.php`
- `2025_08_31_234258_add_channel_tracking_to_loan_disbursements_table.php`
- `2025_08_31_235000_add_bank_relationship_to_loan_offers_table.php`

## Deployment Steps

### Step 1: Deploy Code Changes
```bash
# Pull latest code changes
git pull origin main

# Install dependencies (if any)
composer install --no-dev --optimize-autoloader

# Clear all caches
php artisan cache:clear
php artisan config:clear
php artisan route:clear
php artisan view:clear
```

### Step 2: Run Database Migrations
```bash
# Check migration status
php artisan migrate:status

# Run migrations (with confirmation)
php artisan migrate

# Seed banks data
php artisan db:seed --class=BankSwiftCodeSeeder
```

### Step 3: Update Environment Configuration
Add these to your `.env` file:
```env
# Feature flags for gradual rollout
USE_NORMALIZED_APPROVALS=false
USE_NORMALIZED_DISBURSEMENTS=false
USE_SWIFT_CODE_MAPPING=false

# Start with flags disabled to maintain current behavior
# Enable gradually after testing
```

### Step 4: Test Critical Functions
Run these tests to ensure backward compatibility:

#### Test ESS Integration
```bash
# Send test loan request to ESS endpoint
curl -X POST https://yourdomain.com/api/employee_loan \
  -H "Content-Type: application/xml" \
  -d @test_loan_request.xml
```

#### Test NMB Disbursement
```sql
-- Check recent disbursements
SELECT * FROM loan_offers 
WHERE created_at > DATE_SUB(NOW(), INTERVAL 1 HOUR)
ORDER BY created_at DESC LIMIT 5;
```

### Step 5: Enable Features Gradually

#### Phase 1: Enable SWIFT Code Mapping (Day 1)
```env
USE_SWIFT_CODE_MAPPING=true
```
Monitor for 24 hours, check logs for any issues.

#### Phase 2: Enable Normalized Approvals (Day 3)
```env
USE_NORMALIZED_APPROVALS=true
```
Monitor approval tracking in new tables.

#### Phase 3: Enable Normalized Disbursements (Day 5)
```env
USE_NORMALIZED_DISBURSEMENTS=true
```
Monitor disbursement records.

## Monitoring Queries

### Check Approval Status
```sql
-- View recent approvals
SELECT 
    lo.application_number,
    loa.approval_type,
    loa.status,
    u.name as approved_by,
    loa.created_at
FROM loan_offer_approvals loa
JOIN loan_offers lo ON loa.loan_offer_id = lo.id
LEFT JOIN users u ON loa.approved_by = u.id
WHERE loa.created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)
ORDER BY loa.created_at DESC;
```

### Check Disbursement Channels
```sql
-- Monitor channel distribution
SELECT 
    channel_identifier,
    destination_code,
    COUNT(*) as count,
    SUM(amount) as total_amount
FROM loan_disbursements
WHERE created_at > DATE_SUB(NOW(), INTERVAL 7 DAY)
GROUP BY channel_identifier, destination_code;
```

### Check Bank SWIFT Code Usage
```sql
-- Verify SWIFT code mapping
SELECT 
    b.name as bank_name,
    b.swift_code,
    COUNT(ld.id) as disbursement_count
FROM loan_disbursements ld
JOIN banks b ON ld.bank_id = b.id
WHERE ld.created_at > DATE_SUB(NOW(), INTERVAL 30 DAY)
GROUP BY b.id
ORDER BY disbursement_count DESC;
```

## Rollback Plan

### Quick Rollback (if issues occur)
```bash
# Restore database
mysql -u root -p uraerp < uraerp_backup_YYYYMMDD_HHMMSS.sql

# Revert code changes
git revert HEAD

# Clear caches
php artisan cache:clear
php artisan config:clear
```

### Gradual Rollback (disable features)
```env
# In .env, disable features
USE_NORMALIZED_APPROVALS=false
USE_NORMALIZED_DISBURSEMENTS=false
USE_SWIFT_CODE_MAPPING=false
```

## Verification Tests

### 1. ESS Message Processing
```php
// Test script to verify ESS compatibility
php artisan tinker
>>> $controller = new \App\Http\Controllers\EmployeeLoanController();
>>> $testXml = file_get_contents('test_loan_request.xml');
>>> $response = $controller->handleIncomingMessage($testXml);
>>> echo $response;
```

### 2. NMB Disbursement Test
```php
// Test disbursement with new channel logic
php artisan tinker
>>> $loanOffer = \App\Models\LoanOffer::where('approval_status', 'approved')
    ->whereNull('disbursement_status')->first();
>>> $service = new \App\Services\NmbDisbursementService();
>>> $result = $service->disburseLoan($loanOffer);
>>> print_r($result);
```

## Post-Deployment Monitoring

### Daily Checks (First Week)
1. Check Laravel logs: `tail -f storage/logs/laravel.log`
2. Monitor ESS response times
3. Verify NMB disbursement success rates
4. Check for any null values in new tables

### Weekly Reports
Generate weekly reports on:
- Approval turnaround times
- Disbursement success rates by channel
- Bank distribution for disbursements
- Any failed transactions

## Troubleshooting

### Issue: Migrations fail
```bash
# Check migration status
php artisan migrate:status

# Rollback specific migration
php artisan migrate:rollback --step=1
```

### Issue: ESS messages not processing
```bash
# Check ESS certificate
ls -la /home/crm/ess_utumishi_go_tz.crt

# Verify ESS configuration
php artisan tinker
>>> config('services.ess');
```

### Issue: NMB disbursements failing
```bash
# Check NMB configuration
php artisan tinker
>>> config('services.nmb');

# Test JWT generation
>>> $service = new \App\Services\NmbDisbursementService();
>>> $token = $service->generateJwtToken();
>>> echo $token;
```

## Support Contacts

For urgent issues:
1. Check system logs first
2. Review this deployment guide
3. Contact development team with:
   - Error messages from logs
   - Steps to reproduce issue
   - Time when issue occurred

## Success Metrics

After successful deployment, you should see:
- ✅ All ESS messages processing normally
- ✅ NMB disbursements succeeding with correct channels
- ✅ Approval records created in loan_offer_approvals
- ✅ Disbursement records created in loan_disbursements
- ✅ Bank relationships linked via SWIFT codes
- ✅ No increase in error rates
- ✅ Response times remain stable

---

Last Updated: 2025-09-01
Version: 1.0