<?php
/**
 * Diagnose why bank is not showing in modal
 */

require_once __DIR__ . '/vendor/autoload.php';
$app = require_once __DIR__ . '/bootstrap/app.php';
$kernel = $app->make(Illuminate\Contracts\Http\Kernel::class);
$response = $kernel->handle(
    $request = Illuminate\Http\Request::capture()
);

use App\Models\Bank;
use App\Models\LoanOffer;

echo "\n===========================================\n";
echo "BANK RELATIONSHIP DIAGNOSTIC\n";
echo "===========================================\n\n";

// 1. Check if banks exist
$banksCount = Bank::count();
echo "1. Total banks in database: {$banksCount}\n\n";

// 2. Check TCB specifically
$tcb = Bank::where('swift_code', 'TAPBTZTZ')->first();
if ($tcb) {
    echo "2. TCB Bank found:\n";
    echo "   - ID: {$tcb->id}\n";
    echo "   - Name: {$tcb->name}\n";
    echo "   - Short Name: {$tcb->short_name}\n";
    echo "   - SWIFT: {$tcb->swift_code}\n\n";
} else {
    echo "2. TCB Bank NOT FOUND - Creating it now...\n";
    $tcb = Bank::create([
        'swift_code' => 'TAPBTZTZ',
        'name' => 'TANZANIA COMMERCIAL BANK PLC',
        'short_name' => 'TCB'
    ]);
    echo "   ✓ TCB created with ID: {$tcb->id}\n\n";
}

// 3. Check loans with TAPBTZTZ swift code
echo "3. Loans with TAPBTZTZ swift code:\n";
$loansWithTcbSwift = LoanOffer::where('swift_code', 'TAPBTZTZ')->count();
$loansLinkedToTcb = LoanOffer::where('bank_id', $tcb->id)->count();
$loansUnlinked = LoanOffer::where('swift_code', 'TAPBTZTZ')->whereNull('bank_id')->count();

echo "   - Total with TAPBTZTZ swift: {$loansWithTcbSwift}\n";
echo "   - Linked to TCB bank: {$loansLinkedToTcb}\n";
echo "   - Unlinked: {$loansUnlinked}\n\n";

// 4. Link unlinked loans
if ($loansUnlinked > 0) {
    echo "4. Linking unlinked loans...\n";
    $updated = LoanOffer::where('swift_code', 'TAPBTZTZ')
                        ->whereNull('bank_id')
                        ->update(['bank_id' => $tcb->id]);
    echo "   ✓ Linked {$updated} loans to TCB\n\n";
} else {
    echo "4. All loans are properly linked\n\n";
}

// 5. Test the relationship loading
echo "5. Testing relationship loading (as controller does):\n";
$testLoan = LoanOffer::with('bank')
                     ->where('swift_code', 'TAPBTZTZ')
                     ->first();

if ($testLoan) {
    echo "   Sample loan: #{$testLoan->application_number}\n";
    echo "   - swift_code: {$testLoan->swift_code}\n";
    echo "   - bank_id: {$testLoan->bank_id}\n";
    
    if ($testLoan->bank) {
        echo "   ✓ Bank relationship loaded:\n";
        echo "     - Bank Name: {$testLoan->bank->name}\n";
        echo "     - Short Name: {$testLoan->bank->short_name}\n";
        echo "     - SWIFT: {$testLoan->bank->swift_code}\n";
    } else {
        echo "   ✗ Bank relationship NOT loaded (this is the problem!)\n";
        echo "   Checking why...\n";
        
        if (!$testLoan->bank_id) {
            echo "   - Problem: bank_id is null\n";
            echo "   - Solution: Setting bank_id now...\n";
            $testLoan->bank_id = $tcb->id;
            $testLoan->save();
            echo "   ✓ Fixed!\n";
        } else {
            echo "   - bank_id exists ({$testLoan->bank_id}) but relationship not loading\n";
            echo "   - Check if bank with ID {$testLoan->bank_id} exists...\n";
            $checkBank = Bank::find($testLoan->bank_id);
            if ($checkBank) {
                echo "   - Bank exists, relationship should work\n";
            } else {
                echo "   - Bank with ID {$testLoan->bank_id} doesn't exist!\n";
            }
        }
    }
} else {
    echo "   No loans found with TAPBTZTZ swift code\n";
}

// 6. Show what the blade template will see
echo "\n6. What the blade template sees:\n";
$sampleForBlade = LoanOffer::with('bank')->where('swift_code', 'TAPBTZTZ')->first();
if ($sampleForBlade) {
    $arrayData = $sampleForBlade->toArray();
    echo "   - Loan array has 'bank' key: " . (isset($arrayData['bank']) ? 'YES' : 'NO') . "\n";
    
    if ($sampleForBlade->bank) {
        $bankArray = [
            'id' => $sampleForBlade->bank->id,
            'name' => $sampleForBlade->bank->name,
            'short_name' => $sampleForBlade->bank->short_name,
            'swift_code' => $sampleForBlade->bank->swift_code
        ];
        echo "   - Bank data that would be in JSON:\n";
        echo "     " . json_encode($bankArray, JSON_PRETTY_PRINT) . "\n";
    }
}

echo "\n===========================================\n";
echo "SUMMARY:\n";
echo "===========================================\n";
if ($loansLinkedToTcb > 0 && $testLoan && $testLoan->bank) {
    echo "✓ Everything looks good! Banks should display in modal.\n";
    echo "  Clear browser cache (Ctrl+F5) and try again.\n";
} else {
    echo "✗ Issues found and fixed. Please reload the page.\n";
}
echo "\n";