-- Monitoring Queries for Normalized Loan System
-- Use these queries to monitor the health and performance of the loan system

-- ============================================
-- 1. DAILY MONITORING QUERIES
-- ============================================

-- Check today's loan applications
SELECT 
    COUNT(*) as total_applications,
    SUM(CASE WHEN approval_status = 'approved' THEN 1 ELSE 0 END) as approved,
    SUM(CASE WHEN approval_status = 'rejected' THEN 1 ELSE 0 END) as rejected,
    SUM(CASE WHEN approval_status = 'pending' THEN 1 ELSE 0 END) as pending,
    SUM(loan_amount) as total_loan_amount,
    AVG(loan_amount) as avg_loan_amount
FROM loan_offers
WHERE DATE(created_at) = CURDATE();

-- Check approval workflow status
SELECT 
    loa.approval_type,
    loa.status,
    COUNT(*) as count,
    AVG(TIMESTAMPDIFF(HOUR, loa.created_at, loa.updated_at)) as avg_hours_to_decision
FROM loan_offer_approvals loa
WHERE DATE(loa.created_at) >= DATE_SUB(CURDATE(), INTERVAL 7 DAY)
GROUP BY loa.approval_type, loa.status
ORDER BY loa.approval_type, loa.status;

-- Monitor disbursement performance
SELECT 
    DATE(ld.created_at) as disbursement_date,
    ld.status,
    ld.channel_identifier,
    COUNT(*) as count,
    SUM(ld.amount) as total_amount,
    AVG(TIMESTAMPDIFF(MINUTE, ld.created_at, ld.disbursed_at)) as avg_minutes_to_disburse
FROM loan_disbursements ld
WHERE ld.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
GROUP BY DATE(ld.created_at), ld.status, ld.channel_identifier
ORDER BY disbursement_date DESC, ld.status;

-- ============================================
-- 2. CHANNEL PERFORMANCE ANALYSIS
-- ============================================

-- Channel distribution and success rates
SELECT 
    channel_identifier,
    destination_code,
    COUNT(*) as total_transactions,
    SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) as successful,
    SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed,
    ROUND(SUM(CASE WHEN status = 'success' THEN 1 ELSE 0 END) * 100.0 / COUNT(*), 2) as success_rate,
    SUM(amount) as total_amount,
    AVG(amount) as avg_amount
FROM loan_disbursements
WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
GROUP BY channel_identifier, destination_code
ORDER BY total_transactions DESC;

-- Bank utilization analysis
SELECT 
    b.name as bank_name,
    b.swift_code,
    COUNT(DISTINCT lo.id) as loan_count,
    COUNT(DISTINCT ld.id) as disbursement_count,
    SUM(ld.amount) as total_disbursed,
    AVG(ld.amount) as avg_disbursement
FROM banks b
LEFT JOIN loan_offers lo ON lo.bank_id = b.id
LEFT JOIN loan_disbursements ld ON ld.bank_id = b.id
WHERE lo.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    OR ld.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
GROUP BY b.id
ORDER BY total_disbursed DESC;

-- ============================================
-- 3. ERROR AND EXCEPTION MONITORING
-- ============================================

-- Failed disbursements analysis
SELECT 
    ld.id,
    lo.application_number,
    lo.check_number,
    CONCAT(lo.first_name, ' ', lo.last_name) as customer_name,
    ld.amount,
    ld.channel_identifier,
    ld.destination_code,
    ld.error_message,
    ld.created_at,
    ld.updated_at
FROM loan_disbursements ld
JOIN loan_offers lo ON ld.loan_offer_id = lo.id
WHERE ld.status = 'failed'
    AND ld.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
ORDER BY ld.created_at DESC;

-- Stuck approvals (pending for too long)
SELECT 
    lo.application_number,
    lo.check_number,
    CONCAT(lo.first_name, ' ', lo.last_name) as customer_name,
    loa.approval_type,
    loa.created_at,
    TIMESTAMPDIFF(HOUR, loa.created_at, NOW()) as hours_pending
FROM loan_offer_approvals loa
JOIN loan_offers lo ON loa.loan_offer_id = lo.id
WHERE loa.status = 'pending'
    AND loa.created_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)
ORDER BY hours_pending DESC;

-- ============================================
-- 4. FINANCIAL RECONCILIATION
-- ============================================

-- Daily financial summary
SELECT 
    DATE(created_at) as date,
    COUNT(*) as total_loans,
    SUM(loan_amount) as total_loan_amount,
    SUM(insurance_fee) as total_insurance,
    SUM(processing_fee) as total_processing_fee,
    SUM(total_amount_to_pay) as total_to_disburse,
    SUM(CASE WHEN disbursement_status = 'SUCCESS' THEN total_amount_to_pay ELSE 0 END) as actual_disbursed
FROM loan_offers
WHERE created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
GROUP BY DATE(created_at)
ORDER BY date DESC;

-- Disbursement reconciliation
SELECT 
    lo.application_number,
    lo.total_amount_to_pay as expected_amount,
    ld.amount as disbursed_amount,
    (lo.total_amount_to_pay - ld.amount) as difference,
    ld.status,
    ld.disbursed_at
FROM loan_offers lo
LEFT JOIN loan_disbursements ld ON ld.loan_offer_id = lo.id
WHERE lo.approval_status = 'approved'
    AND lo.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
    AND ABS(lo.total_amount_to_pay - COALESCE(ld.amount, 0)) > 0.01
ORDER BY difference DESC;

-- ============================================
-- 5. TOP-UP LOANS MONITORING
-- ============================================

-- Top-up loan statistics
SELECT 
    COUNT(*) as total_topups,
    SUM(lo_new.loan_amount) as total_new_loan_amount,
    SUM(lot.settlement_amount) as total_settlement_amount,
    SUM(lot.top_up_amount) as total_topup_amount,
    AVG(lot.top_up_amount) as avg_topup_amount
FROM loan_offer_topups lot
JOIN loan_offers lo_new ON lot.new_loan_id = lo_new.id
WHERE lot.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY);

-- Top-up loan details
SELECT 
    lo_original.application_number as original_loan,
    lo_new.application_number as new_loan,
    lot.settlement_amount,
    lot.top_up_amount,
    lo_new.loan_amount as new_loan_amount,
    lo_new.approval_status,
    lot.created_at
FROM loan_offer_topups lot
JOIN loan_offers lo_original ON lot.original_loan_id = lo_original.id
JOIN loan_offers lo_new ON lot.new_loan_id = lo_new.id
WHERE lot.created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY)
ORDER BY lot.created_at DESC;

-- ============================================
-- 6. USER ACTIVITY MONITORING
-- ============================================

-- Approval activity by user
SELECT 
    u.name as user_name,
    loa.approval_type,
    COUNT(*) as total_decisions,
    SUM(CASE WHEN loa.status = 'approved' THEN 1 ELSE 0 END) as approved,
    SUM(CASE WHEN loa.status = 'rejected' THEN 1 ELSE 0 END) as rejected,
    AVG(TIMESTAMPDIFF(MINUTE, loa.created_at, loa.updated_at)) as avg_decision_time_minutes
FROM loan_offer_approvals loa
JOIN users u ON loa.approved_by = u.id OR loa.rejected_by = u.id
WHERE loa.updated_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
    AND loa.status != 'pending'
GROUP BY u.id, loa.approval_type
ORDER BY total_decisions DESC;

-- ============================================
-- 7. SYSTEM HEALTH CHECKS
-- ============================================

-- Check for orphaned records
SELECT 'Approvals without loan offers' as issue, COUNT(*) as count
FROM loan_offer_approvals loa
LEFT JOIN loan_offers lo ON loa.loan_offer_id = lo.id
WHERE lo.id IS NULL
UNION ALL
SELECT 'Disbursements without loan offers' as issue, COUNT(*) as count
FROM loan_disbursements ld
LEFT JOIN loan_offers lo ON ld.loan_offer_id = lo.id
WHERE lo.id IS NULL
UNION ALL
SELECT 'Top-ups without original loans' as issue, COUNT(*) as count
FROM loan_offer_topups lot
LEFT JOIN loan_offers lo ON lot.original_loan_id = lo.id
WHERE lo.id IS NULL;

-- Check data consistency
SELECT 
    'Approved loans without disbursement records' as check_type,
    COUNT(*) as count
FROM loan_offers lo
WHERE lo.approval_status = 'approved'
    AND lo.disbursement_status = 'SUCCESS'
    AND NOT EXISTS (
        SELECT 1 FROM loan_disbursements ld 
        WHERE ld.loan_offer_id = lo.id
    )
    AND lo.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY);

-- ============================================
-- 8. PERFORMANCE METRICS
-- ============================================

-- Average processing times by stage
SELECT 
    'Application to Approval' as stage,
    AVG(TIMESTAMPDIFF(HOUR, lo.created_at, loa.updated_at)) as avg_hours
FROM loan_offers lo
JOIN loan_offer_approvals loa ON loa.loan_offer_id = lo.id
WHERE loa.status = 'approved'
    AND lo.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY)
UNION ALL
SELECT 
    'Approval to Disbursement' as stage,
    AVG(TIMESTAMPDIFF(HOUR, loa.updated_at, ld.disbursed_at)) as avg_hours
FROM loan_offer_approvals loa
JOIN loan_disbursements ld ON ld.loan_offer_id = loa.loan_offer_id
WHERE loa.status = 'approved'
    AND ld.status = 'success'
    AND loa.created_at >= DATE_SUB(NOW(), INTERVAL 30 DAY);

-- ============================================
-- 9. ALERT QUERIES (Run every hour)
-- ============================================

-- High-value disbursements alert
SELECT 
    lo.application_number,
    CONCAT(lo.first_name, ' ', lo.last_name) as customer_name,
    ld.amount,
    ld.channel_identifier,
    b.name as bank_name,
    ld.created_at
FROM loan_disbursements ld
JOIN loan_offers lo ON ld.loan_offer_id = lo.id
LEFT JOIN banks b ON ld.bank_id = b.id
WHERE ld.amount >= 50000000  -- Alert for disbursements >= 50M TZS
    AND ld.created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
ORDER BY ld.amount DESC;

-- Failed disbursement rate alert
SELECT 
    COUNT(*) as total_disbursements,
    SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) as failed_count,
    ROUND(SUM(CASE WHEN status = 'failed' THEN 1 ELSE 0 END) * 100.0 / COUNT(*), 2) as failure_rate
FROM loan_disbursements
WHERE created_at >= DATE_SUB(NOW(), INTERVAL 1 HOUR)
HAVING failure_rate > 10;  -- Alert if failure rate > 10%

-- ============================================
-- 10. MONTHLY REPORTING QUERIES
-- ============================================

-- Monthly loan portfolio summary
SELECT 
    DATE_FORMAT(created_at, '%Y-%m') as month,
    COUNT(*) as total_applications,
    SUM(CASE WHEN approval_status = 'approved' THEN 1 ELSE 0 END) as approved_loans,
    SUM(CASE WHEN disbursement_status = 'SUCCESS' THEN 1 ELSE 0 END) as disbursed_loans,
    SUM(loan_amount) as total_loan_amount,
    SUM(CASE WHEN disbursement_status = 'SUCCESS' THEN total_amount_to_pay ELSE 0 END) as total_disbursed,
    AVG(loan_amount) as avg_loan_amount,
    MIN(loan_amount) as min_loan_amount,
    MAX(loan_amount) as max_loan_amount
FROM loan_offers
WHERE created_at >= DATE_SUB(NOW(), INTERVAL 12 MONTH)
GROUP BY DATE_FORMAT(created_at, '%Y-%m')
ORDER BY month DESC;