<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use App\Models\User;
use App\Models\Enquiry;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Cache;
use Carbon\Carbon;

class CheckOverloadedUsers extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'users:check-overloaded';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Check for users with more than 20 pending assigned enquiries and send SMS alerts to their HODs every 3 hours';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $this->info('Checking for overloaded users...');

        // Check Loan Officers
        $this->checkLoanOfficers();

        // Check Accountants
        $this->checkAccountants();

        $this->info('Overload check completed successfully.');

        return 0;
    }

    /**
     * Check loan officers with overload
     */
    private function checkLoanOfficers()
    {
        $loanOfficers = User::whereHas('roles', function ($q) {
            $q->where('name', 'loanofficer');
        })->get();

        $overloadedOfficers = [];

        foreach ($loanOfficers as $officer) {
            $assignedCount = Enquiry::where('type', 'loan_application')
                ->where('status', 'assigned')
                ->whereHas('assignedUsers', function ($q) use ($officer) {
                    $q->where('user_id', $officer->id);
                })
                ->count();

            if ($assignedCount > 20) {
                $overloadedOfficers[] = [
                    'user' => $officer,
                    'count' => $assignedCount
                ];

                $this->warn("Loan Officer {$officer->name} has {$assignedCount} pending enquiries");
            }
        }

        if (count($overloadedOfficers) > 0) {
            $this->sendAlertToHOD('head_of_department_loan', $overloadedOfficers, 'Loan');
        }
    }

    /**
     * Check accountants with overload
     */
    private function checkAccountants()
    {
        $accountants = User::whereHas('roles', function ($q) {
            $q->where('name', 'accountant');
        })->get();

        $overloadedAccountants = [];

        foreach ($accountants as $accountant) {
            $assignedCount = Enquiry::where('type', '!=', 'loan_application')
                ->where('status', 'assigned')
                ->whereHas('assignedUsers', function ($q) use ($accountant) {
                    $q->where('user_id', $accountant->id);
                })
                ->count();

            if ($assignedCount > 20) {
                $overloadedAccountants[] = [
                    'user' => $accountant,
                    'count' => $assignedCount
                ];

                $this->warn("Accountant {$accountant->name} has {$assignedCount} pending enquiries");
            }
        }

        if (count($overloadedAccountants) > 0) {
            $this->sendAlertToHOD('head_of_department_accountant', $overloadedAccountants, 'Accountant');
        }
    }

    /**
     * Send alert to Head of Department
     */
    private function sendAlertToHOD($roleName, $overloadedUsers, $departmentName)
    {
        // Get HOD
        $hod = User::whereHas('roles', function ($q) use ($roleName) {
            $q->where('name', $roleName);
        })->first();

        if (!$hod || empty($hod->phone_number)) {
            $this->error("No HOD found or phone number not set for {$roleName}");
            return;
        }

        // Check if we've sent alert in last 3 hours
        $cacheKey = "overload_alert_{$roleName}_" . now()->format('Y-m-d_H');

        if (Cache::has($cacheKey)) {
            $this->info("Alert already sent to {$departmentName} HOD in the last 3 hours");
            return;
        }

        // Build SMS message
        $message = "URASACCOS ALERT:\n\n";
        $message .= "The following {$departmentName} staff have more than 20 pending assigned enquiries:\n\n";

        foreach ($overloadedUsers as $item) {
            $message .= "- {$item['user']->name}: {$item['count']} pending\n";
        }

        $message .= "\nPlease review and reassign workload as needed.";

        // Send SMS
        $sent = $this->sendSMS($hod->phone_number, $message);

        if ($sent) {
            // Cache for 3 hours
            Cache::put($cacheKey, true, now()->addHours(3));
            $this->info("Alert sent successfully to {$departmentName} HOD: {$hod->name}");
        } else {
            $this->error("Failed to send alert to {$departmentName} HOD");
        }
    }

    /**
     * Send SMS using API
     */
    private function sendSMS($phoneNumber, $message)
    {
        try {
            // Clean phone number
            $phone = preg_replace('/[^0-9]/', '', $phoneNumber);

            // Ensure it starts with 255
            if (substr($phone, 0, 1) === '0') {
                $phone = '255' . substr($phone, 1);
            } elseif (substr($phone, 0, 3) !== '255') {
                $phone = '255' . $phone;
            }

            // Get SMS credentials from env
            $apiKey = env('SMS_API_KEY');
            $secretKey = env('SMS_SECRET_KEY');
            $sourceAddr = env('SMS_SOURCE_ADDR', 'URASACCOS');

            if (!$apiKey || !$secretKey) {
                $this->error('SMS API credentials not configured');
                return false;
            }

            // Send SMS via API
            $response = Http::post('https://apisms.beem.africa/public/v1/send', [
                'source_addr' => $sourceAddr,
                'encoding' => 0,
                'schedule_time' => '',
                'message' => $message,
                'recipients' => [
                    [
                        'recipient_id' => 1,
                        'dest_addr' => $phone
                    ]
                ]
            ], [
                'Authorization' => 'Basic ' . base64_encode($apiKey . ':' . $secretKey),
                'Content-Type' => 'application/json'
            ]);

            return $response->successful();

        } catch (\Exception $e) {
            $this->error('SMS sending error: ' . $e->getMessage());
            return false;
        }
    }
}
